<?php

namespace Modules\Flowmaker\Models\Nodes;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Modules\Flowmaker\Models\Contact;
use Modules\Flowmaker\Models\Traits\InterpolatesVariables;

class NewRequest extends Node
{
    use InterpolatesVariables;

    public function process($message, $data)
    {
        Log::info('Processing message in NewRequest node', ['message' => $message, 'data' => $data]);
        try {
            $settings = $this->getDataAsArray()['settings']['new_request'] ?? [];
            Log::info('NewRequest Settings', ['settings' => $settings]);

            $contactId = is_object($data) ? $data->contact_id : $data['contact_id'];
            $contact = Contact::find($contactId);
            if (!$contact) {
                Log::error('Contact not found', ['contactId' => $contactId]);
                return ['success' => false];
            }

            $method = strtoupper($settings['method'] ?? 'GET');
            $url = $this->interpolate($settings['url'] ?? '', $contact, $this->flow_id);
            $headersJson = $settings['headers'] ?? '{}';
            $bodyJson = $settings['body'] ?? '{}';
            $bodyType = $settings['bodyType'] ?? 'json';
            $responseVar = $settings['responseVar'] ?? '';
            $queryParams = $settings['queryParams'] ?? [];
            $files = $settings['files'] ?? [];
            $conditions = $settings['conditions'] ?? [];
            $enableConditionalRouting = $settings['enableConditionalRouting'] ?? false;
            $pagination = $settings['pagination'] ?? [];
            $enablePagination = $pagination['enableSendMore'] ?? false;

            if (empty($url)) {
                Log::error('URL is empty after interpolation');
                return ['success' => false];
            }

            // Add query parameters to URL
            if (!empty($queryParams)) {
                $url = $this->addQueryParamsToUrl($url, $queryParams, $contact);
            }

            // Parse headers JSON
            $headers = [];
            try {
                $headers = json_decode($headersJson, true) ?: [];
            } catch (\Exception $e) {
                Log::error('Invalid headers JSON', ['error' => $e->getMessage()]);
            }

            // Interpolate variables in headers
            foreach ($headers as $k => $v) {
                $headers[$k] = $this->interpolate($v, $contact, $this->flow_id);
            }

            // Process body based on body type
            $body = $this->processBody($bodyJson, $bodyType, $contact, $files);

            $timeout = isset($settings['timeout']) ? (int)$settings['timeout'] : 10; // default 10s
            $retries = isset($settings['retry']) ? (int)$settings['retry'] : 0; // default 0
            $http = Http::timeout($timeout);
            if ($retries > 0) {
                $http = $http->retry($retries, 100);
            }

            Log::info('Making HTTP request', [
                'method' => $method,
                'url' => $url,
                'headers' => $headers,
                'body' => $body,
                'bodyType' => $bodyType,
                'timeout' => $timeout,
                'retries' => $retries
            ]);

            // Make HTTP request
            $response = null;
            switch ($method) {
                case 'GET':
                    $response = $http->withHeaders($headers)->get($url, $body);
                    break;
                case 'POST':
                    $response = $this->makePostRequest($http, $url, $headers, $body, $bodyType);
                    break;
                case 'PUT':
                    $response = $this->makePutRequest($http, $url, $headers, $body, $bodyType);
                    break;
                case 'PATCH':
                    $response = $this->makePatchRequest($http, $url, $headers, $body, $bodyType);
                    break;
                case 'DELETE':
                    $response = $http->withHeaders($headers)->delete($url, $body);
                    break;
                case 'HEAD':
                    $response = $http->withHeaders($headers)->head($url, $body);
                    break;
                case 'OPTIONS':
                    $response = $http->withHeaders($headers)->send('OPTIONS', $url, ['body' => $body]);
                    break;
                default:
                    Log::error('Unsupported HTTP method', ['method' => $method]);
                    return ['success' => false];
            }

            if ($response) {
                $contentType = $response->header('Content-Type');
                $json = null;
                if (strpos($contentType, 'xml') !== false) {
                    // Parse XML response
                    $xml = simplexml_load_string($response->body());
                    $json = json_decode(json_encode($xml), true);
                } else {
                    $json = $response->json();
                }
                $body = $response->body();
                $formatted = "🛒 Namah Demo Product List:\n";
                $maxLen = 3900; // Stay under WhatsApp's 4096 char limit
                $count = 1;
                if (is_array($json)) {
                    foreach ($json as $item) {
                        $line = $count . ". ";
                        $line .= isset($item['name']) ? $item['name'] : 'Unknown';
                        $details = [];
                        if (isset($item['data']) && is_array($item['data'])) {
                            if (isset($item['data']['price']) || isset($item['data']['Price'])) {
                                $price = $item['data']['price'] ?? $item['data']['Price'];
                                $details[] = "$" . $price;
                            }
                            if (isset($item['data']['color']) || isset($item['data']['Color'])) {
                                $color = $item['data']['color'] ?? $item['data']['Color'];
                                $details[] = $color;
                            }
                            if (isset($item['data']['capacity']) || isset($item['data']['Capacity']) || isset($item['data']['capacity GB']) || isset($item['data']['Capacity GB'])) {
                                $capacity = $item['data']['capacity'] ?? $item['data']['Capacity'] ?? $item['data']['capacity GB'] ?? $item['data']['Capacity GB'];
                                $details[] = $capacity;
                            }
                        }
                        if (!empty($details)) {
                            $line .= " - " . implode(", ", $details);
                        }
                        $formatted .= $line . "\n";
                        $count++;
                        if (strlen($formatted) > $maxLen) {
                            $formatted .= "...and more.";
                            break;
                        }
                    }
                } else {
                    $formatted .= "No products found.";
                }

                // Handle conditional routing
                if ($enableConditionalRouting && !empty($conditions)) {
                    $nextNodeId = $this->evaluateConditions($conditions, $response, $json);
                    if ($nextNodeId) {
                        return $nextNodeId;
                    }
                }

                // Handle pagination
                if ($enablePagination) {
                    $this->handlePagination($pagination, $response, $json, $contact);
                }

                // Store formatted string in response
                $responseData = [
                    'status' => $response->status(),
                    'formatted' => $formatted,
                    'headers' => $response->headers(),
                ];
                // Save to contact state/context
                $contact->setContactState($this->flow_id, $settings['responseVar'] . '_formatted', $formatted);
                $contact->setContactState($this->flow_id, $settings['responseVar'], json_encode($responseData));
                Log::info('NewRequest node formatted response', ['formatted' => $formatted]);
                return $this->getNextNodeId($data);
            } else {
                Log::error('HTTP request failed - no response');
                return ['success' => false];
            }
        } catch (\Exception $e) {
            Log::error('Error processing NewRequest node', ['error' => $e->getMessage()]);
            return ['success' => false];
        }

        // Continue flow to next node if one exists
        $nextNode = $this->getNextNodeId();
        if ($nextNode) {
            $nextNode->process($message, $data);
        }
        return ['success' => true];
    }

    /**
     * Add query parameters to URL
     */
    private function addQueryParamsToUrl($url, $queryParams, $contact)
    {
        $parsedUrl = parse_url($url);
        $existingParams = [];
        if (isset($parsedUrl['query'])) {
            parse_str($parsedUrl['query'], $existingParams);
        }

        foreach ($queryParams as $param) {
            if (!empty($param['key'])) {
                $key = $this->interpolate($param['key'], $contact, $this->flow_id);
                $value = $this->interpolate($param['value'], $contact, $this->flow_id);
                $existingParams[$key] = $value;
            }
        }

        $parsedUrl['query'] = http_build_query($existingParams);
        return $this->buildUrl($parsedUrl);
    }

    /**
     * Build URL from parsed components
     */
    private function buildUrl($parsedUrl)
    {
        $url = '';
        if (isset($parsedUrl['scheme'])) {
            $url .= $parsedUrl['scheme'] . '://';
        }
        if (isset($parsedUrl['host'])) {
            $url .= $parsedUrl['host'];
        }
        if (isset($parsedUrl['port'])) {
            $url .= ':' . $parsedUrl['port'];
        }
        if (isset($parsedUrl['path'])) {
            $url .= $parsedUrl['path'];
        }
        if (isset($parsedUrl['query'])) {
            $url .= '?' . $parsedUrl['query'];
        }
        return $url;
    }

    /**
     * Process body based on body type
     */
    private function processBody($bodyJson, $bodyType, $contact, $files = [])
    {
        switch ($bodyType) {
            case 'json':
                try {
                    $body = json_decode($bodyJson, true) ?: [];
                    // Interpolate variables in body
                    foreach ($body as $k => $v) {
                        $body[$k] = $this->interpolate($v, $contact, $this->flow_id);
                    }
                    return $body;
                } catch (\Exception $e) {
                    Log::error('Invalid JSON body', ['error' => $e->getMessage()]);
                    return [];
                }

            case 'form':
                // Parse form data (key=value format)
                $formData = [];
                $lines = explode("\n", $bodyJson);
                foreach ($lines as $line) {
                    $line = trim($line);
                    if (empty($line)) continue;
                    
                    $parts = explode('=', $line, 2);
                    if (count($parts) === 2) {
                        $key = trim($parts[0]);
                        $value = trim($parts[1]);
                        $formData[$key] = $this->interpolate($value, $contact, $this->flow_id);
                    }
                }
                return $formData;

            case 'xml':
                // Return raw XML string with interpolated variables
                return $this->interpolate($bodyJson, $contact, $this->flow_id);

            case 'multipart':
                // Handle multipart with files
                $formData = [];
                $lines = explode("\n", $bodyJson);
                foreach ($lines as $line) {
                    $line = trim($line);
                    if (empty($line)) continue;
                    
                    $parts = explode('=', $line, 2);
                    if (count($parts) === 2) {
                        $key = trim($parts[0]);
                        $value = trim($parts[1]);
                        $formData[$key] = $this->interpolate($value, $contact, $this->flow_id);
                    }
                }

                // Add files to form data
                foreach ($files as $file) {
                    if (!empty($file['fieldName'])) {
                        $formData[$file['fieldName']] = $file['file'] ?? null;
                    }
                }
                return $formData;

            default:
                return [];
        }
    }

    /**
     * Make POST request with appropriate body handling
     */
    private function makePostRequest($http, $url, $headers, $body, $bodyType)
    {
        switch ($bodyType) {
            case 'json':
                return $http->withHeaders($headers)->post($url, $body);
            case 'form':
                return $http->withHeaders($headers)->asForm()->post($url, $body);
            case 'xml':
                $headers['Content-Type'] = 'application/xml';
                return $http->withHeaders($headers)->withBody($body, 'application/xml')->post($url);
            case 'multipart':
                return $http->withHeaders($headers)->asForm()->post($url, $body);
            default:
                return $http->withHeaders($headers)->post($url, $body);
        }
    }

    /**
     * Make PUT request with appropriate body handling
     */
    private function makePutRequest($http, $url, $headers, $body, $bodyType)
    {
        switch ($bodyType) {
            case 'json':
                return $http->withHeaders($headers)->put($url, $body);
            case 'form':
                return $http->withHeaders($headers)->asForm()->put($url, $body);
            case 'xml':
                $headers['Content-Type'] = 'application/xml';
                return $http->withHeaders($headers)->withBody($body, 'application/xml')->put($url);
            case 'multipart':
                return $http->withHeaders($headers)->asForm()->put($url, $body);
            default:
                return $http->withHeaders($headers)->put($url, $body);
        }
    }

    /**
     * Make PATCH request with appropriate body handling
     */
    private function makePatchRequest($http, $url, $headers, $body, $bodyType)
    {
        switch ($bodyType) {
            case 'json':
                return $http->withHeaders($headers)->patch($url, $body);
            case 'form':
                return $http->withHeaders($headers)->asForm()->patch($url, $body);
            case 'xml':
                $headers['Content-Type'] = 'application/xml';
                return $http->withHeaders($headers)->withBody($body, 'application/xml')->patch($url);
            case 'multipart':
                return $http->withHeaders($headers)->asForm()->patch($url, $body);
            default:
                return $http->withHeaders($headers)->patch($url, $body);
        }
    }

    /**
     * Evaluate conditions for conditional routing
     */
    private function evaluateConditions($conditions, $response, $json)
    {
        foreach ($conditions as $condition) {
            $type = $condition['type'] ?? '';
            $operator = $condition['operator'] ?? '';
            $value = $condition['value'] ?? '';
            $targetNodeId = $condition['targetNodeId'] ?? null;

            $result = false;
            switch ($type) {
                case 'status_code':
                    $result = $this->evaluateStatusCondition($response->status(), $operator, $value);
                    break;
                case 'response_contains':
                    $result = $this->evaluateContainsCondition($response->body(), $operator, $value);
                    break;
                case 'response_json':
                    $result = $this->evaluateJsonCondition($json, $operator, $value);
                    break;
            }

            if ($result && $targetNodeId) {
                return $targetNodeId;
            }
        }
        return null;
    }

    /**
     * Evaluate status code condition
     */
    private function evaluateStatusCondition($status, $operator, $value)
    {
        switch ($operator) {
            case 'equals':
                return $status == $value;
            case 'not_equals':
                return $status != $value;
            case 'greater_than':
                return $status > $value;
            case 'less_than':
                return $status < $value;
            default:
                return false;
        }
    }

    /**
     * Evaluate contains condition
     */
    private function evaluateContainsCondition($body, $operator, $value)
    {
        switch ($operator) {
            case 'contains':
                return strpos($body, $value) !== false;
            case 'not_contains':
                return strpos($body, $value) === false;
            case 'starts_with':
                return strpos($body, $value) === 0;
            case 'ends_with':
                return substr($body, -strlen($value)) === $value;
            default:
                return false;
        }
    }

    /**
     * Evaluate JSON condition
     */
    private function evaluateJsonCondition($json, $operator, $value)
    {
        switch ($operator) {
            case 'exists':
                return isset($json[$value]);
            case 'not_exists':
                return !isset($json[$value]);
            case 'equals':
                return isset($json[$value]) && $json[$value] == $value;
            case 'not_equals':
                return isset($json[$value]) && $json[$value] != $value;
            default:
                return false;
        }
    }

    /**
     * Handle pagination
     */
    private function handlePagination($pagination, $response, $json, $contact)
    {
        $nextPagePath = $pagination['nextPagePath'] ?? '';
        $hasMorePath = $pagination['hasMorePath'] ?? '';
        $maxPages = $pagination['maxPages'] ?? 5;

        // Store pagination info in contact state
        $currentPage = $contact->getContactStateValue($this->flow_id, 'current_page') ?? 1;
        
        if ($currentPage < $maxPages) {
            $hasMore = false;
            if (!empty($hasMorePath) && isset($json[$hasMorePath])) {
                $hasMore = $json[$hasMorePath];
            } elseif (!empty($nextPagePath) && isset($json[$nextPagePath])) {
                $hasMore = !empty($json[$nextPagePath]);
            }

            if ($hasMore) {
                $contact->setContactState($this->flow_id, 'current_page', $currentPage + 1);
                $contact->setContactState($this->flow_id, 'has_more_data', true);
            } else {
                $contact->setContactState($this->flow_id, 'has_more_data', false);
            }
        } else {
            $contact->setContactState($this->flow_id, 'has_more_data', false);
        }
    }

    protected function getNextNodeId($data = null)
    {
        // Get the first outgoing edge's target
        if (!empty($this->outgoingEdges)) {
            return $this->outgoingEdges[0]->getTarget();
        }
        return null;
    }
} 