<?php

namespace Modules\Flowmaker\Models\Nodes;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Modules\Flowmaker\Models\Contact;
use Modules\Flowmaker\Services\SMTPValidator;
use Modules\Flowmaker\Services\SMTPConfigManager;
use Modules\Flowmaker\Services\EmailErrorHandler;
use Illuminate\Mail\Mailable;
use Illuminate\Mail\Mailables\Content;
use Illuminate\Mail\Mailables\Envelope;
use Illuminate\Mail\Mailables\Attachment;
use Symfony\Component\Mailer\Transport\Smtp\EsmtpTransport;

class SendEmail extends Node
{
    private SMTPValidator $smtpValidator;
    private SMTPConfigManager $configManager;
    private EmailErrorHandler $errorHandler;

    public function __construct()
    {
        parent::__construct();
        $this->smtpValidator = new SMTPValidator();
        $this->configManager = new SMTPConfigManager();
        $this->errorHandler = new EmailErrorHandler();
    }

    public function process($message, $data)
    {
        Log::info('🔄 Processing SendEmail node', [
            'node_id' => $this->id ?? 'unknown',
            'flow_id' => $this->flow_id ?? 'unknown'
        ]);

        try {
            // Load settings
            $nodeData = $this->getDataAsArray();
            $settings = $nodeData['settings'] ?? [];
            Log::debug('📋 SendEmail settings', ['settings' => $settings]);

            // Resolve contact
            $contactId = is_object($data) ? ($data->contact_id ?? null) : ($data['contact_id'] ?? null);
            $contact = $contactId ? Contact::find($contactId) : null;
            if (!$contact) {
                Log::error('❌ Contact not found', ['contactId' => $contactId]);
                return ['success' => false];
            }

            // Extract settings - Initialize all variables first
            $to = '';
            $cc = '';
            $bcc = '';
            $subject = '';
            $body = '';
            $isHtml = true; // Default to HTML
            $fromEmail = '';
            $fromName = '';
            $responseVar = '';
            $attachments = [];
            $smtpConfig = [];
            
            // Now assign values
            if (isset($settings['to'])) {
                $to = $settings['to'];
            }
            if (isset($settings['cc'])) {
                $cc = $settings['cc'];
            }
            if (isset($settings['bcc'])) {
                $bcc = $settings['bcc'];
            }
            if (isset($settings['subject'])) {
                $subject = $settings['subject'];
            }
            if (isset($settings['body'])) {
                $body = $settings['body'];
            }
            if (isset($settings['isHtml'])) {
                $isHtml = (bool) $settings['isHtml'];
            }
            if (isset($settings['fromEmail'])) {
                $fromEmail = $settings['fromEmail'];
            }
            if (isset($settings['fromName'])) {
                $fromName = $settings['fromName'];
            }
            if (isset($settings['responseVar'])) {
                $responseVar = trim($settings['responseVar']);
            }
            if (isset($settings['attachments'])) {
                $attachments = $settings['attachments'];
            }
            if (isset($settings['smtpConfig'])) {
                $smtpConfig = $settings['smtpConfig'];
            }

            // Safety check - ensure all variables are defined
            if (!isset($to)) {
                Log::error('❌ Variable $to is not defined after initialization');
                return ['success' => false, 'error' => 'Variable initialization failed'];
            }
            
            // Debug: Log initial settings
            Log::debug('📧 Initial email settings', [
                'to' => $to,
                'subject' => $subject,
                'fromEmail' => $fromEmail,
                'settings_keys' => array_keys($settings),
                'all_variables_defined' => [
                    'to' => isset($to),
                    'cc' => isset($cc),
                    'bcc' => isset($bcc),
                    'subject' => isset($subject),
                    'body' => isset($body),
                    'fromEmail' => isset($fromEmail),
                    'fromName' => isset($fromName)
                ]
            ]);

            // Interpolate variables
            $originalTo = $to;
            $to = $contact->changeVariables($to, $this->flow_id);
            $cc = $contact->changeVariables($cc, $this->flow_id);
            $bcc = $contact->changeVariables($bcc, $this->flow_id);
            $subject = $contact->changeVariables($subject, $this->flow_id);
            $body = $contact->changeVariables($body, $this->flow_id);
            $fromEmail = $contact->changeVariables($fromEmail, $this->flow_id);
            $fromName = $contact->changeVariables($fromName, $this->flow_id);

            Log::debug('📧 Email variables after interpolation', [
                'original_to' => $originalTo,
                'interpolated_to' => $to,
                'subject' => $subject,
                'fromEmail' => $fromEmail,
                'fromName' => $fromName
            ]);

            if (empty($to)) {
                Log::error('❌ Recipient email is empty', [
                    'original_to' => $originalTo,
                    'interpolated_to' => $to,
                    'settings' => $settings
                ]);
                return ['success' => false, 'error' => 'Recipient email is required'];
            }

            // Create custom mailable
            $mailable = new class($subject, $body, $isHtml, $fromEmail, $fromName, $attachments) extends Mailable {
                public $subject;
                public $body;
                public $isHtml;
                public $fromEmail;
                public $fromName;
                public $attachments;

                public function __construct($subject, $body, $isHtml, $fromEmail, $fromName, $attachments)
                {
                    $this->subject = $subject;
                    $this->body = $body;
                    $this->isHtml = $isHtml;
                    $this->fromEmail = $fromEmail;
                    $this->fromName = $fromName;
                    $this->attachments = $attachments;
                }


                public function build()
                {
                    $mail = $this->subject($this->subject);
                    
                    if (!empty($this->fromEmail)) {
                        $mail->from($this->fromEmail, $this->fromName);
                    }
                    
                    if ($this->isHtml) {
                        $mail->html($this->body);
                    } else {
                        // For plain text, use html() with stripped tags
                        $mail->html(nl2br(strip_tags($this->body)));
                    }
                    
                    return $mail;
                }

                public function attachments()
                {
                    $attachments = [];
                    foreach ($this->attachments as $attachment) {
                        if (isset($attachment['path']) && file_exists($attachment['path'])) {
                            $attachments[] = Attachment::fromPath($attachment['path'])
                                ->as($attachment['name'] ?? basename($attachment['path']));
                        }
                    }
                    return $attachments;
                }
            };

            // Add CC and BCC to mailable if provided
            if (!empty($cc)) {
                $ccEmails = array_map('trim', explode(',', $cc));
                $mailable->cc($ccEmails);
            }
            
            if (!empty($bcc)) {
                $bccEmails = array_map('trim', explode(',', $bcc));
                $mailable->bcc($bccEmails);
            }

            // Attempt to send email with validation and fallback
            $emailResult = $this->attemptEmailSendWithFallback($settings, $mailable, $to);
            
            if (!$emailResult['success']) {
                // Store failure in response variable if specified
                if ($responseVar !== '') {
                    $contact->setContactState($this->flow_id, $responseVar, json_encode($emailResult));
                }
                return $emailResult;
            }

            $result = $emailResult['result'];

            // Store response in contact state
            if ($responseVar !== '') {
                $response = [
                    'success' => true,
                    'sent_at' => now()->toISOString(),
                    'to' => $to,
                    'subject' => $subject,
                    'cc' => $cc,
                    'bcc' => $bcc
                ];
                $contact->setContactState($this->flow_id, $responseVar, json_encode($response));
                Log::info('✅ Stored email response in contact state', ['variable' => $responseVar]);
            }

            // Continue to next node
            $nextNode = $this->getNextNodeId();
            if ($nextNode) {
                $nextNode->process($message, $data);
            }

            return ['success' => true, 'result' => $result];

        } catch (\Exception $e) {
            $errorResponse = $this->errorHandler->handleSMTPError($e, [
                'node_id' => $this->id ?? 'unknown',
                'flow_id' => $this->flow_id ?? 'unknown'
            ]);
            
            Log::error('❌ SendEmail node error', [
                'node_id' => $this->id ?? 'unknown',
                'flow_id' => $this->flow_id ?? 'unknown',
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            
            return $errorResponse;
        }
    }

    /**
     * Attempt to send email with SMTP validation and fallback logic
     */
    protected function attemptEmailSendWithFallback(array $settings, $mailable, string $to): array
    {
        // Try custom SMTP first if configured
        $customConfig = $this->configManager->getCustomConfig($settings);
        if (!empty($customConfig)) {
            Log::info('🔍 Attempting email send with custom SMTP configuration');
            $this->errorHandler->logEmailAttempt('custom', false, 'Starting attempt');
            
            $customResult = $this->attemptEmailSend($customConfig, $mailable, $to, 'custom');
            if ($customResult['success']) {
                $this->errorHandler->logEmailAttempt('custom', true);
                return $customResult;
            }
            
            Log::warning('⚠️ Custom SMTP failed, attempting fallback to default configuration');
            $this->errorHandler->logEmailAttempt('custom', false, $customResult['error']);
        }

        // Fallback to default SMTP configuration
        $defaultConfig = $this->configManager->getDefaultConfig();
        if (!empty($defaultConfig)) {
            Log::info('🔍 Attempting email send with default SMTP configuration');
            $this->errorHandler->logEmailAttempt('default', false, 'Starting fallback attempt');
            
            $defaultResult = $this->attemptEmailSend($defaultConfig, $mailable, $to, 'default');
            if ($defaultResult['success']) {
                $this->errorHandler->logEmailAttempt('default', true);
                return $defaultResult;
            }
            
            $this->errorHandler->logEmailAttempt('default', false, $defaultResult['error']);
        }

        // Both configurations failed
        return $this->errorHandler->createErrorResponse(
            'All SMTP configurations failed. Please check your email settings.',
            [
                'attempted_configs' => array_filter(['custom' => !empty($customConfig), 'default' => !empty($defaultConfig)]),
                'custom_error' => $customResult['error'] ?? 'Not attempted',
                'default_error' => $defaultResult['error'] ?? 'Not attempted'
            ]
        );
    }

    /**
     * Attempt to send email with specific SMTP configuration
     */
    protected function attemptEmailSend(array $smtpConfig, $mailable, string $to, string $configType = 'unknown'): array
    {
        try {
            // Validate SMTP configuration first
            if (!$this->validateSMTPConfig($smtpConfig)) {
                return $this->errorHandler->createErrorResponse(
                    'SMTP configuration validation failed',
                    ['config_type' => $configType, 'validator_error' => $this->smtpValidator->getLastError()]
                );
            }

            // Apply SMTP configuration
            $this->configManager->switchToConfig($smtpConfig);

            Log::debug('📧 Sending email with validated configuration', [
                'config_type' => $configType,
                'to' => $to,
                'host' => $smtpConfig['host'],
                'port' => $smtpConfig['port']
            ]);

            // Send the email
            $mailer = Mail::mailer();
            $result = $mailer->to($to)->send($mailable);

            // Log success
            $this->errorHandler->logSMTPSuccess($smtpConfig, 'send');

            return $this->errorHandler->createSuccessResponse([
                'config_used' => $configType,
                'to' => $to,
                'sent_at' => now()->toISOString(),
                'smtp_host' => $smtpConfig['host']
            ]);

        } catch (\Exception $e) {
            // Reset configuration on failure
            $this->configManager->resetToDefault();
            
            return $this->errorHandler->handleSMTPError($e, array_merge($smtpConfig, [
                'config_type' => $configType
            ]));
        }
    }

    /**
     * Validate SMTP configuration before attempting to send
     */
    protected function validateSMTPConfig(array $smtpConfig): bool
    {
        Log::debug('🔍 Validating SMTP configuration', [
            'host' => $smtpConfig['host'] ?? 'missing',
            'port' => $smtpConfig['port'] ?? 'missing',
            'username' => $smtpConfig['username'] ?? 'missing'
        ]);

        $validationResult = $this->smtpValidator->validateCredentials($smtpConfig);
        
        if (!$validationResult->isValid) {
            Log::error('❌ SMTP validation failed', [
                'error' => $validationResult->errorMessage,
                'response_time' => $validationResult->responseTime,
                'tested_config' => $validationResult->testedConfig
            ]);
            return false;
        }

        Log::info('✅ SMTP validation successful', [
            'host' => $smtpConfig['host'],
            'response_time' => round($validationResult->responseTime, 3) . 's'
        ]);

        return true;
    }

    protected function getNextNodeId($data = null)
    {
        if (!empty($this->outgoingEdges)) {
            return $this->outgoingEdges[0]->getTarget();
        }
        return null;
    }
}