<?php

namespace Modules\Flowmaker\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\RateLimiter;

class EmailSecurityService
{
    private const VALIDATION_CACHE_TTL = 300; // 5 minutes
    private const RATE_LIMIT_KEY_PREFIX = 'email_send_';
    private const MAX_EMAILS_PER_MINUTE = 10;
    private const MAX_EMAILS_PER_HOUR = 100;

    /**
     * Sanitize SMTP configuration for logging (remove sensitive data)
     */
    public function sanitizeConfigForLogging(array $config): array
    {
        $sanitized = $config;
        
        // Remove or mask sensitive fields
        $sensitiveFields = ['password', 'api_key', 'secret', 'token'];
        
        foreach ($sensitiveFields as $field) {
            if (isset($sanitized[$field])) {
                $sanitized[$field] = $this->maskSensitiveValue($sanitized[$field]);
            }
        }

        return $sanitized;
    }

    /**
     * Validate email address format and security
     */
    public function validateEmailAddress(string $email): array
    {
        $errors = [];

        // Basic format validation
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'Invalid email format';
        }

        // Check for suspicious patterns
        if ($this->containsSuspiciousPatterns($email)) {
            $errors[] = 'Email contains suspicious patterns';
        }

        // Check domain reputation (basic check)
        if ($this->isDisposableEmailDomain($email)) {
            $errors[] = 'Disposable email addresses are not allowed';
        }

        return $errors;
    }

    /**
     * Check rate limiting for email sending
     */
    public function checkRateLimit(string $identifier): array
    {
        $minuteKey = self::RATE_LIMIT_KEY_PREFIX . 'minute_' . $identifier;
        $hourKey = self::RATE_LIMIT_KEY_PREFIX . 'hour_' . $identifier;

        // Check per-minute limit
        if (RateLimiter::tooManyAttempts($minuteKey, self::MAX_EMAILS_PER_MINUTE)) {
            return [
                'allowed' => false,
                'reason' => 'Too many emails sent per minute',
                'retry_after' => RateLimiter::availableIn($minuteKey)
            ];
        }

        // Check per-hour limit
        if (RateLimiter::tooManyAttempts($hourKey, self::MAX_EMAILS_PER_HOUR)) {
            return [
                'allowed' => false,
                'reason' => 'Too many emails sent per hour',
                'retry_after' => RateLimiter::availableIn($hourKey)
            ];
        }

        // Increment counters
        RateLimiter::hit($minuteKey, 60); // 1 minute
        RateLimiter::hit($hourKey, 3600); // 1 hour

        return ['allowed' => true];
    }

    /**
     * Cache SMTP validation results to improve performance
     */
    public function getCachedValidationResult(array $smtpConfig): ?array
    {
        $cacheKey = $this->generateValidationCacheKey($smtpConfig);
        return Cache::get($cacheKey);
    }

    /**
     * Store SMTP validation result in cache
     */
    public function cacheValidationResult(array $smtpConfig, array $result): void
    {
        $cacheKey = $this->generateValidationCacheKey($smtpConfig);
        
        // Only cache successful validations
        if ($result['isValid'] ?? false) {
            Cache::put($cacheKey, $result, self::VALIDATION_CACHE_TTL);
        }
    }

    /**
     * Validate SSL/TLS configuration
     */
    public function validateSSLConfig(array $smtpConfig): array
    {
        $errors = [];
        $encryption = $smtpConfig['encryption'] ?? 'tls';
        $port = $smtpConfig['port'] ?? 587;

        // Check port/encryption compatibility
        if ($encryption === 'ssl' && $port !== 465) {
            $errors[] = 'SSL encryption typically uses port 465';
        }

        if ($encryption === 'tls' && !in_array($port, [587, 25])) {
            $errors[] = 'TLS encryption typically uses port 587 or 25';
        }

        // Validate SSL context options
        if (isset($smtpConfig['verify_peer']) && !is_bool($smtpConfig['verify_peer'])) {
            $errors[] = 'verify_peer must be a boolean value';
        }

        return $errors;
    }

    /**
     * Secure timeout management
     */
    public function validateTimeout(int $timeout): array
    {
        $errors = [];

        if ($timeout < 5) {
            $errors[] = 'Timeout too short (minimum 5 seconds)';
        }

        if ($timeout > 120) {
            $errors[] = 'Timeout too long (maximum 120 seconds)';
        }

        return $errors;
    }

    /**
     * Log security events
     */
    public function logSecurityEvent(string $event, array $context = []): void
    {
        Log::warning('🔒 Email Security Event', array_merge([
            'event' => $event,
            'timestamp' => now()->toISOString(),
            'ip' => request()->ip() ?? 'unknown'
        ], $context));
    }

    /**
     * Mask sensitive values for logging
     */
    private function maskSensitiveValue(string $value): string
    {
        if (strlen($value) <= 4) {
            return '***';
        }

        return substr($value, 0, 2) . str_repeat('*', strlen($value) - 4) . substr($value, -2);
    }

    /**
     * Check for suspicious patterns in email addresses
     */
    private function containsSuspiciousPatterns(string $email): bool
    {
        $suspiciousPatterns = [
            '/[<>"\']/',           // HTML/script injection characters
            '/javascript:/i',       // JavaScript protocol
            '/data:/i',            // Data protocol
            '/\.\.//',             // Directory traversal
            '/\x00/',              // Null bytes
        ];

        foreach ($suspiciousPatterns as $pattern) {
            if (preg_match($pattern, $email)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if email domain is a known disposable email provider
     */
    private function isDisposableEmailDomain(string $email): bool
    {
        $domain = strtolower(substr(strrchr($email, '@'), 1));
        
        // Common disposable email domains
        $disposableDomains = [
            '10minutemail.com',
            'tempmail.org',
            'guerrillamail.com',
            'mailinator.com',
            'throwaway.email',
            'temp-mail.org'
        ];

        return in_array($domain, $disposableDomains);
    }

    /**
     * Generate cache key for SMTP validation
     */
    private function generateValidationCacheKey(array $smtpConfig): string
    {
        // Create cache key from non-sensitive config data
        $keyData = [
            'host' => $smtpConfig['host'] ?? '',
            'port' => $smtpConfig['port'] ?? '',
            'encryption' => $smtpConfig['encryption'] ?? '',
            'username' => $smtpConfig['username'] ?? ''
        ];

        return 'smtp_validation_' . md5(serialize($keyData));
    }

    /**
     * Clean up old cache entries
     */
    public function cleanupValidationCache(): void
    {
        // This would typically be called by a scheduled job
        $pattern = 'smtp_validation_*';
        
        // Note: This is a simplified cleanup - in production you might want
        // to use a more sophisticated cache tagging system
        Log::info('🧹 Cleaning up SMTP validation cache');
    }

    /**
     * Get rate limit status for an identifier
     */
    public function getRateLimitStatus(string $identifier): array
    {
        $minuteKey = self::RATE_LIMIT_KEY_PREFIX . 'minute_' . $identifier;
        $hourKey = self::RATE_LIMIT_KEY_PREFIX . 'hour_' . $identifier;

        return [
            'minute' => [
                'attempts' => RateLimiter::attempts($minuteKey),
                'max' => self::MAX_EMAILS_PER_MINUTE,
                'remaining' => max(0, self::MAX_EMAILS_PER_MINUTE - RateLimiter::attempts($minuteKey)),
                'reset_in' => RateLimiter::availableIn($minuteKey)
            ],
            'hour' => [
                'attempts' => RateLimiter::attempts($hourKey),
                'max' => self::MAX_EMAILS_PER_HOUR,
                'remaining' => max(0, self::MAX_EMAILS_PER_HOUR - RateLimiter::attempts($hourKey)),
                'reset_in' => RateLimiter::availableIn($hourKey)
            ]
        ];
    }
}