<?php

namespace Modules\Flowmaker\Tests\Integration;

use Tests\TestCase;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Config;
use Modules\Flowmaker\Models\Nodes\SendEmail;
use Modules\Flowmaker\Models\Contact;
use Modules\Flowmaker\Services\SMTPValidator;
use Modules\Flowmaker\Services\SMTPConfigManager;
use Modules\Flowmaker\Services\EmailErrorHandler;

class EmailSendingIntegrationTest extends TestCase
{
    use RefreshDatabase;

    private SendEmail $sendEmailNode;
    private Contact $testContact;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create test contact
        $this->testContact = Contact::factory()->create([
            'email' => 'test@example.com',
            'name' => 'Test Contact'
        ]);

        // Create SendEmail node
        $this->sendEmailNode = new SendEmail();
        $this->sendEmailNode->flow_id = 1;
        $this->sendEmailNode->id = 'test-email-node';
    }

    /** @test */
    public function it_sends_email_with_valid_default_smtp_configuration()
    {
        Mail::fake();

        // Configure default SMTP
        Config::set('mail.default', 'smtp');
        Config::set('mail.mailers.smtp', [
            'transport' => 'smtp',
            'host' => 'localhost',
            'port' => 1025,
            'username' => 'test@local.com',
            'password' => 'password',
            'encryption' => 'tls'
        ]);

        $settings = [
            'to' => 'recipient@example.com',
            'subject' => 'Test Email',
            'body' => 'This is a test email',
            'fromEmail' => 'sender@example.com',
            'fromName' => 'Test Sender',
            'responseVar' => 'email_result',
            'smtpConfig' => [
                'useCustom' => false
            ]
        ];

        // Mock the node data
        $this->sendEmailNode->shouldReceive('getDataAsArray')
            ->andReturn(['settings' => $settings]);

        $messageData = (object) [
            'contact_id' => $this->testContact->id,
            'value' => 'test message'
        ];

        $result = $this->sendEmailNode->process('test message', $messageData);

        $this->assertTrue($result['success']);
        Mail::assertSent(\Illuminate\Mail\Mailable::class);
    }

    /** @test */
    public function it_falls_back_to_default_smtp_when_custom_fails()
    {
        Mail::fake();

        // Configure valid default SMTP
        Config::set('mail.default', 'smtp');
        Config::set('mail.mailers.smtp', [
            'transport' => 'smtp',
            'host' => 'localhost',
            'port' => 1025,
            'username' => 'test@local.com',
            'password' => 'password',
            'encryption' => 'tls'
        ]);

        $settings = [
            'to' => 'recipient@example.com',
            'subject' => 'Test Email',
            'body' => 'This is a test email',
            'fromEmail' => 'sender@example.com',
            'fromName' => 'Test Sender',
            'responseVar' => 'email_result',
            'smtpConfig' => [
                'useCustom' => true,
                'host' => 'invalid.smtp.server',
                'port' => 587,
                'username' => 'invalid@example.com',
                'password' => 'wrongpassword',
                'encryption' => 'tls',
                'timeout' => 5
            ]
        ];

        $this->sendEmailNode->shouldReceive('getDataAsArray')
            ->andReturn(['settings' => $settings]);

        $messageData = (object) [
            'contact_id' => $this->testContact->id,
            'value' => 'test message'
        ];

        // This should fail custom SMTP but succeed with default
        $result = $this->sendEmailNode->process('test message', $messageData);

        // Should eventually succeed with fallback
        $this->assertTrue($result['success']);
        Mail::assertSent(\Illuminate\Mail\Mailable::class);
    }

    /** @test */
    public function it_handles_complete_smtp_failure_gracefully()
    {
        Mail::fake();

        // Configure invalid default SMTP
        Config::set('mail.default', 'smtp');
        Config::set('mail.mailers.smtp', [
            'transport' => 'smtp',
            'host' => 'invalid.server.com',
            'port' => 9999,
            'username' => 'invalid@example.com',
            'password' => 'wrongpassword'
        ]);

        $settings = [
            'to' => 'recipient@example.com',
            'subject' => 'Test Email',
            'body' => 'This is a test email',
            'responseVar' => 'email_result',
            'smtpConfig' => [
                'useCustom' => true,
                'host' => 'another.invalid.server',
                'port' => 587,
                'username' => 'invalid@example.com',
                'password' => 'wrongpassword',
                'encryption' => 'tls',
                'timeout' => 5
            ]
        ];

        $this->sendEmailNode->shouldReceive('getDataAsArray')
            ->andReturn(['settings' => $settings]);

        $messageData = (object) [
            'contact_id' => $this->testContact->id,
            'value' => 'test message'
        ];

        $result = $this->sendEmailNode->process('test message', $messageData);

        $this->assertFalse($result['success']);
        $this->assertArrayHasKey('error', $result);
        $this->assertArrayHasKey('details', $result);
    }

    /** @test */
    public function it_validates_smtp_configuration_before_sending()
    {
        $validator = new SMTPValidator();
        
        // Test with invalid configuration
        $invalidConfig = [
            'host' => '',
            'port' => 'invalid',
            'username' => 'test@example.com',
            'password' => 'password'
        ];

        $result = $validator->validateCredentials($invalidConfig);
        
        $this->assertFalse($result->isValid);
        $this->assertNotEmpty($result->errorMessage);
    }

    /** @test */
    public function it_manages_smtp_configuration_switching()
    {
        $configManager = new SMTPConfigManager();
        
        $customConfig = [
            'host' => 'smtp.example.com',
            'port' => 587,
            'username' => 'test@example.com',
            'password' => 'password',
            'encryption' => 'tls'
        ];

        // Test switching to custom config
        $configManager->switchToConfig($customConfig);
        
        $currentConfig = config('mail.mailers.smtp');
        $this->assertEquals('smtp.example.com', $currentConfig['host']);
        $this->assertEquals(587, $currentConfig['port']);

        // Test reset to default
        $configManager->resetToDefault();
        
        // Should not throw any errors
        $this->assertTrue(true);
    }

    /** @test */
    public function it_handles_email_errors_with_proper_categorization()
    {
        $errorHandler = new EmailErrorHandler();
        
        // Test authentication error
        $authException = new \Exception('535 Authentication failed');
        $config = ['host' => 'smtp.example.com', 'username' => 'test@example.com'];
        
        $result = $errorHandler->handleSMTPError($authException, $config);
        
        $this->assertFalse($result['success']);
        $this->assertEquals('SMTP_AUTH_FAILED', $result['details']['code']);
        $this->assertArrayHasKey('suggestions', $result['details']);
        $this->assertNotEmpty($result['details']['suggestions']);
    }

    /** @test */
    public function it_interpolates_variables_in_email_content()
    {
        Mail::fake();

        // Configure valid SMTP
        Config::set('mail.default', 'smtp');
        Config::set('mail.mailers.smtp', [
            'transport' => 'smtp',
            'host' => 'localhost',
            'port' => 1025,
            'username' => 'test@local.com',
            'password' => 'password'
        ]);

        $settings = [
            'to' => '{{contact_email}}',
            'subject' => 'Hello {{contact_name}}',
            'body' => 'Dear {{contact_name}}, this is your email.',
            'fromEmail' => 'sender@example.com',
            'responseVar' => 'email_result',
            'smtpConfig' => [
                'useCustom' => false
            ]
        ];

        $this->sendEmailNode->shouldReceive('getDataAsArray')
            ->andReturn(['settings' => $settings]);

        // Mock contact variable interpolation
        $this->testContact->shouldReceive('changeVariables')
            ->with('{{contact_email}}', 1)
            ->andReturn('test@example.com');
        
        $this->testContact->shouldReceive('changeVariables')
            ->with('Hello {{contact_name}}', 1)
            ->andReturn('Hello Test Contact');
            
        $this->testContact->shouldReceive('changeVariables')
            ->with('Dear {{contact_name}}, this is your email.', 1)
            ->andReturn('Dear Test Contact, this is your email.');

        $messageData = (object) [
            'contact_id' => $this->testContact->id,
            'value' => 'test message'
        ];

        $result = $this->sendEmailNode->process('test message', $messageData);

        $this->assertTrue($result['success']);
    }

    /** @test */
    public function it_stores_email_response_in_contact_state()
    {
        Mail::fake();

        Config::set('mail.default', 'smtp');
        Config::set('mail.mailers.smtp', [
            'transport' => 'smtp',
            'host' => 'localhost',
            'port' => 1025,
            'username' => 'test@local.com',
            'password' => 'password'
        ]);

        $settings = [
            'to' => 'recipient@example.com',
            'subject' => 'Test Email',
            'body' => 'Test body',
            'responseVar' => 'email_response',
            'smtpConfig' => [
                'useCustom' => false
            ]
        ];

        $this->sendEmailNode->shouldReceive('getDataAsArray')
            ->andReturn(['settings' => $settings]);

        // Mock contact state setting
        $this->testContact->shouldReceive('setContactState')
            ->with(1, 'email_response', \Mockery::type('string'))
            ->once();

        $messageData = (object) [
            'contact_id' => $this->testContact->id,
            'value' => 'test message'
        ];

        $result = $this->sendEmailNode->process('test message', $messageData);

        $this->assertTrue($result['success']);
    }

    /** @test */
    public function it_logs_smtp_attempts_and_results()
    {
        Log::shouldReceive('info')->atLeast()->once();
        Log::shouldReceive('debug')->atLeast()->once();
        Log::shouldReceive('error')->zeroOrMoreTimes();

        $errorHandler = new EmailErrorHandler();
        
        // Test logging successful attempt
        $errorHandler->logEmailAttempt('custom', true);
        
        // Test logging failed attempt
        $errorHandler->logEmailAttempt('default', false, 'Connection failed');
        
        // Test logging SMTP success
        $config = ['host' => 'smtp.example.com', 'port' => 587];
        $errorHandler->logSMTPSuccess($config, 'send');

        $this->assertTrue(true); // If we get here, logging worked
    }
}