<?php

namespace Modules\Flowmaker\Tests\Unit;

use Tests\TestCase;
use Modules\Flowmaker\Services\EmailErrorHandler;
use Exception;

class EmailErrorHandlerTest extends TestCase
{
    private EmailErrorHandler $errorHandler;

    protected function setUp(): void
    {
        parent::setUp();
        $this->errorHandler = new EmailErrorHandler();
    }

    /** @test */
    public function it_handles_authentication_failure_error()
    {
        $exception = new Exception('535 Authentication failed');
        $config = [
            'host' => 'smtp.gmail.com',
            'port' => 587,
            'username' => 'test@gmail.com'
        ];

        $result = $this->errorHandler->handleSMTPError($exception, $config);

        $this->assertFalse($result['success']);
        $this->assertStringContains('authentication failed', strtolower($result['error']));
        $this->assertEquals('SMTP_AUTH_FAILED', $result['details']['code']);
        $this->assertArrayHasKey('suggestions', $result['details']);
    }

    /** @test */
    public function it_handles_connection_refused_error()
    {
        $exception = new Exception('Connection refused');
        $config = ['host' => 'smtp.example.com', 'port' => 587];

        $result = $this->errorHandler->handleSMTPError($exception, $config);

        $this->assertEquals('SMTP_CONNECTION_REFUSED', $result['details']['code']);
        $this->assertStringContains('refused', strtolower($result['error']));
    }

    /** @test */
    public function it_handles_timeout_error()
    {
        $exception = new Exception('Connection timed out');
        $config = ['host' => 'smtp.example.com'];

        $result = $this->errorHandler->handleSMTPError($exception, $config);

        $this->assertEquals('SMTP_TIMEOUT', $result['details']['code']);
        $this->assertStringContains('timeout', strtolower($result['error']));
    }

    /** @test */
    public function it_handles_ssl_error()
    {
        $exception = new Exception('SSL certificate verification failed');
        $config = ['host' => 'smtp.example.com'];

        $result = $this->errorHandler->handleSMTPError($exception, $config);

        $this->assertEquals('SMTP_SSL_ERROR', $result['details']['code']);
        $this->assertStringContains('ssl', strtolower($result['error']));
    }

    /** @test */
    public function it_creates_error_response_with_context()
    {
        $message = 'Test error message';
        $context = ['key' => 'value', 'number' => 123];

        $result = $this->errorHandler->createErrorResponse($message, $context);

        $this->assertFalse($result['success']);
        $this->assertEquals($message, $result['error']);
        $this->assertEquals($context, $result['details']);
        $this->assertArrayHasKey('timestamp', $result);
    }

    /** @test */
    public function it_creates_success_response()
    {
        $details = ['email_id' => '12345', 'recipient' => 'test@example.com'];

        $result = $this->errorHandler->createSuccessResponse($details);

        $this->assertTrue($result['success']);
        $this->assertEquals('Email sent successfully', $result['message']);
        $this->assertEquals($details, $result['details']);
        $this->assertArrayHasKey('timestamp', $result);
    }

    /** @test */
    public function it_provides_suggestions_for_auth_failed()
    {
        $exception = new Exception('535 Authentication failed');
        $config = ['host' => 'smtp.gmail.com'];

        $result = $this->errorHandler->handleSMTPError($exception, $config);
        $suggestions = $result['details']['suggestions'];

        $this->assertIsArray($suggestions);
        $this->assertNotEmpty($suggestions);
        $this->assertStringContains('password', strtolower(implode(' ', $suggestions)));
        $this->assertStringContains('username', strtolower(implode(' ', $suggestions)));
    }

    /** @test */
    public function it_provides_suggestions_for_connection_refused()
    {
        $exception = new Exception('Connection refused');
        $config = ['host' => 'smtp.example.com'];

        $result = $this->errorHandler->handleSMTPError($exception, $config);
        $suggestions = $result['details']['suggestions'];

        $this->assertIsArray($suggestions);
        $this->assertStringContains('port', strtolower(implode(' ', $suggestions)));
        $this->assertStringContains('host', strtolower(implode(' ', $suggestions)));
    }

    /** @test */
    public function it_sanitizes_config_in_error_response()
    {
        $exception = new Exception('Test error');
        $config = [
            'host' => 'smtp.example.com',
            'username' => 'test@example.com',
            'password' => 'secret123'
        ];

        $result = $this->errorHandler->handleSMTPError($exception, $config);

        // Password should not be exposed in the response
        $this->assertStringNotContains('secret123', json_encode($result));
        $this->assertEquals('test@example.com', $result['details']['username']);
    }

    /** @test */
    public function it_handles_generic_errors()
    {
        $exception = new Exception('Some unknown SMTP error');
        $config = ['host' => 'smtp.example.com'];

        $result = $this->errorHandler->handleSMTPError($exception, $config);

        $this->assertEquals('SMTP_GENERIC_ERROR', $result['details']['code']);
        $this->assertStringContains('Some unknown SMTP error', $result['error']);
    }

    /** @test */
    public function it_includes_original_error_in_details()
    {
        $originalError = 'Original detailed error message';
        $exception = new Exception($originalError);
        $config = ['host' => 'smtp.example.com'];

        $result = $this->errorHandler->handleSMTPError($exception, $config);

        $this->assertEquals($originalError, $result['details']['original_error']);
    }

    /** @test */
    public function it_includes_timestamp_in_responses()
    {
        $exception = new Exception('Test error');
        $config = ['host' => 'smtp.example.com'];

        $result = $this->errorHandler->handleSMTPError($exception, $config);

        $this->assertArrayHasKey('timestamp', $result);
        $this->assertNotEmpty($result['timestamp']);
        
        // Should be a valid ISO timestamp
        $timestamp = \DateTime::createFromFormat(\DateTime::ATOM, $result['timestamp']);
        $this->assertInstanceOf(\DateTime::class, $timestamp);
    }
}