import React, { useState } from 'react';
import { Handle, Position, NodeProps } from '@xyflow/react';
import { Plug, Settings, Play, Database } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';

interface FlowiseNodeProps {
  data: any;
  id: string;
  selected?: boolean;
  isConnectable?: boolean;
}

const FlowiseNode: React.FC<FlowiseNodeProps> = ({ data, selected }) => {
  const [isEditing, setIsEditing] = useState(false);
  const [settings, setSettings] = useState(data.settings || {
    apiUrl: '',
    apiKey: '',
    chatflowId: '',
    message: '',
    sessionId: '',
    variables: [],
    timeout: 30,
    retryCount: 3,
    temperature: 0.7,
    maxTokens: 1000,
    responseVar: 'flowise_response'
  });

  const handleSave = () => {
    data.settings = settings;
    setIsEditing(false);
  };

  const addVariable = () => {
    setSettings(prev => ({
      ...prev,
      variables: [...prev.variables, { key: '', value: '' }]
    }));
  };

  const updateVariable = (index: number, field: 'key' | 'value', value: string) => {
    setSettings(prev => ({
      ...prev,
      variables: prev.variables.map((v, i) => 
        i === index ? { ...v, [field]: value } : v
      )
    }));
  };

  const removeVariable = (index: number) => {
    setSettings(prev => ({
      ...prev,
      variables: prev.variables.filter((_, i) => i !== index)
    }));
  };

  return (
    <div className={`px-4 py-2 shadow-md rounded-md bg-indigo-100 border-2 min-w-[300px] ${
      selected ? 'border-indigo-500' : 'border-indigo-200'
    }`}>
      <Handle type="target" position={Position.Top} className="w-3 h-3" />
      
      <div className="flex items-center gap-2 mb-2">
        <Plug className="w-4 h-4 text-indigo-600" />
        <div className="font-bold text-indigo-800">FlowiseAI</div>
        <Button
          variant="ghost"
          size="sm"
          onClick={() => setIsEditing(!isEditing)}
          className="ml-auto p-1 h-6 w-6"
        >
          <Settings className="w-3 h-3" />
        </Button>
      </div>

      {isEditing ? (
        <div className="space-y-3">
          <div>
            <Label htmlFor="apiUrl">API URL</Label>
            <Input
              id="apiUrl"
              value={settings.apiUrl}
              onChange={(e) => setSettings(prev => ({ ...prev, apiUrl: e.target.value }))}
              placeholder="https://your-flowise-instance.com"
            />
          </div>

          <div>
            <Label htmlFor="apiKey">API Key</Label>
            <Input
              id="apiKey"
              value={settings.apiKey}
              onChange={(e) => setSettings(prev => ({ ...prev, apiKey: e.target.value }))}
              placeholder="Enter Flowise API key"
              type="password"
            />
          </div>

          <div>
            <Label htmlFor="chatflowId">Chatflow ID</Label>
            <Input
              id="chatflowId"
              value={settings.chatflowId}
              onChange={(e) => setSettings(prev => ({ ...prev, chatflowId: e.target.value }))}
              placeholder="Enter Chatflow ID"
            />
          </div>

          <div>
            <Label htmlFor="sessionId">Session ID</Label>
            <Input
              id="sessionId"
              value={settings.sessionId}
              onChange={(e) => setSettings(prev => ({ ...prev, sessionId: e.target.value }))}
              placeholder="Enter Session ID (optional)"
            />
          </div>

          <div>
            <Label htmlFor="message">Message</Label>
            <Textarea
              id="message"
              value={settings.message}
              onChange={(e) => setSettings(prev => ({ ...prev, message: e.target.value }))}
              placeholder="Enter message to send to Flowise"
              rows={3}
            />
          </div>

          <div>
            <Label>Variables</Label>
            {settings.variables.map((variable, index) => (
              <div key={index} className="flex gap-2 mb-2">
                <Input
                  value={variable.key}
                  onChange={(e) => updateVariable(index, 'key', e.target.value)}
                  placeholder="Key"
                />
                <Input
                  value={variable.value}
                  onChange={(e) => updateVariable(index, 'value', e.target.value)}
                  placeholder="Value"
                />
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => removeVariable(index)}
                  className="px-2"
                >
                  ×
                </Button>
              </div>
            ))}
            <Button variant="outline" size="sm" onClick={addVariable} className="w-full">
              Add Variable
            </Button>
          </div>

          <div className="grid grid-cols-2 gap-2">
            <div>
              <Label htmlFor="timeout">Timeout (seconds)</Label>
              <Input
                id="timeout"
                type="number"
                value={settings.timeout}
                onChange={(e) => setSettings(prev => ({ ...prev, timeout: parseInt(e.target.value) || 30 }))}
              />
            </div>
            <div>
              <Label htmlFor="retryCount">Retry Count</Label>
              <Input
                id="retryCount"
                type="number"
                value={settings.retryCount}
                onChange={(e) => setSettings(prev => ({ ...prev, retryCount: parseInt(e.target.value) || 3 }))}
              />
            </div>
          </div>

          <div className="grid grid-cols-2 gap-2">
            <div>
              <Label htmlFor="temperature">Temperature</Label>
              <Input
                id="temperature"
                type="number"
                step="0.1"
                min="0"
                max="2"
                value={settings.temperature}
                onChange={(e) => setSettings(prev => ({ ...prev, temperature: parseFloat(e.target.value) || 0.7 }))}
              />
            </div>
            <div>
              <Label htmlFor="maxTokens">Max Tokens</Label>
              <Input
                id="maxTokens"
                type="number"
                value={settings.maxTokens}
                onChange={(e) => setSettings(prev => ({ ...prev, maxTokens: parseInt(e.target.value) || 1000 }))}
              />
            </div>
          </div>

          <div>
            <Label htmlFor="responseVar">Response Variable</Label>
            <Input
              id="responseVar"
              value={settings.responseVar}
              onChange={(e) => setSettings(prev => ({ ...prev, responseVar: e.target.value }))}
              placeholder="Variable name to store Flowise response"
            />
          </div>

          <div className="flex gap-2">
            <Button onClick={handleSave} size="sm" className="flex-1">
              Save
            </Button>
            <Button 
              variant="outline" 
              onClick={() => setIsEditing(false)} 
              size="sm" 
              className="flex-1"
            >
              Cancel
            </Button>
          </div>
        </div>
      ) : (
        <div className="text-sm text-indigo-700">
          <div className="mb-1">
            <strong>API URL:</strong> {settings.apiUrl ? new URL(settings.apiUrl).hostname : 'Not configured'}
          </div>
          <div className="mb-1">
            <strong>Chatflow:</strong> {settings.chatflowId || 'Not configured'}
          </div>
          <div className="mb-1">
            <strong>Session:</strong> {settings.sessionId || 'Auto-generated'}
          </div>
          {settings.message && (
            <div className="mb-1">
              <strong>Message:</strong> {settings.message.substring(0, 50)}
              {settings.message.length > 50 ? '...' : ''}
            </div>
          )}
          {settings.variables.length > 0 && (
            <div className="mb-1">
              <strong>Variables:</strong> {settings.variables.length} configured
            </div>
          )}
          <div className="flex gap-4 text-xs">
            <span>Temp: {settings.temperature}</span>
            <span>Tokens: {settings.maxTokens}</span>
          </div>
        </div>
      )}

      <Handle type="source" position={Position.Bottom} className="w-3 h-3" />
    </div>
  );
};

export default FlowiseNode;
