<?php

namespace Modules\Flowmaker\Tests\Unit;

use Tests\TestCase;
use Modules\Flowmaker\Services\EmailSecurityService;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\RateLimiter;

class EmailSecurityServiceTest extends TestCase
{
    private EmailSecurityService $securityService;

    protected function setUp(): void
    {
        parent::setUp();
        $this->securityService = new EmailSecurityService();
    }

    /** @test */
    public function it_sanitizes_smtp_config_for_logging()
    {
        $config = [
            'host' => 'smtp.gmail.com',
            'port' => 587,
            'username' => 'test@gmail.com',
            'password' => 'secret123',
            'api_key' => 'api_secret_key'
        ];

        $sanitized = $this->securityService->sanitizeConfigForLogging($config);

        $this->assertEquals('smtp.gmail.com', $sanitized['host']);
        $this->assertEquals('test@gmail.com', $sanitized['username']);
        $this->assertNotEquals('secret123', $sanitized['password']);
        $this->assertNotEquals('api_secret_key', $sanitized['api_key']);
        $this->assertStringContains('*', $sanitized['password']);
    }

    /** @test */
    public function it_validates_email_addresses()
    {
        // Valid email
        $errors = $this->securityService->validateEmailAddress('test@example.com');
        $this->assertEmpty($errors);

        // Invalid format
        $errors = $this->securityService->validateEmailAddress('invalid-email');
        $this->assertNotEmpty($errors);
        $this->assertStringContains('Invalid email format', implode(' ', $errors));

        // Suspicious patterns
        $errors = $this->securityService->validateEmailAddress('test<script>@example.com');
        $this->assertNotEmpty($errors);
    }

    /** @test */
    public function it_detects_disposable_email_domains()
    {
        $errors = $this->securityService->validateEmailAddress('test@10minutemail.com');
        $this->assertNotEmpty($errors);
        $this->assertStringContains('Disposable email', implode(' ', $errors));

        $errors = $this->securityService->validateEmailAddress('test@gmail.com');
        $this->assertEmpty($errors);
    }

    /** @test */
    public function it_enforces_rate_limiting()
    {
        $identifier = 'test_user_123';

        // First few attempts should be allowed
        for ($i = 0; $i < 5; $i++) {
            $result = $this->securityService->checkRateLimit($identifier);
            $this->assertTrue($result['allowed']);
        }

        // Mock rate limiter to simulate too many attempts
        RateLimiter::shouldReceive('tooManyAttempts')
            ->andReturn(true);
        RateLimiter::shouldReceive('availableIn')
            ->andReturn(60);

        $result = $this->securityService->checkRateLimit($identifier);
        $this->assertFalse($result['allowed']);
        $this->assertArrayHasKey('reason', $result);
        $this->assertArrayHasKey('retry_after', $result);
    }

    /** @test */
    public function it_validates_ssl_configuration()
    {
        // Valid SSL config
        $config = [
            'encryption' => 'ssl',
            'port' => 465
        ];
        $errors = $this->securityService->validateSSLConfig($config);
        $this->assertEmpty($errors);

        // Invalid SSL config (wrong port)
        $config = [
            'encryption' => 'ssl',
            'port' => 587
        ];
        $errors = $this->securityService->validateSSLConfig($config);
        $this->assertNotEmpty($errors);

        // Valid TLS config
        $config = [
            'encryption' => 'tls',
            'port' => 587
        ];
        $errors = $this->securityService->validateSSLConfig($config);
        $this->assertEmpty($errors);
    }

    /** @test */
    public function it_validates_timeout_values()
    {
        // Valid timeout
        $errors = $this->securityService->validateTimeout(30);
        $this->assertEmpty($errors);

        // Too short
        $errors = $this->securityService->validateTimeout(3);
        $this->assertNotEmpty($errors);
        $this->assertStringContains('too short', implode(' ', $errors));

        // Too long
        $errors = $this->securityService->validateTimeout(150);
        $this->assertNotEmpty($errors);
        $this->assertStringContains('too long', implode(' ', $errors));
    }

    /** @test */
    public function it_caches_validation_results()
    {
        $smtpConfig = [
            'host' => 'smtp.gmail.com',
            'port' => 587,
            'username' => 'test@gmail.com'
        ];

        // No cached result initially
        $cached = $this->securityService->getCachedValidationResult($smtpConfig);
        $this->assertNull($cached);

        // Cache a successful result
        $result = ['isValid' => true, 'message' => 'Success'];
        $this->securityService->cacheValidationResult($smtpConfig, $result);

        // Should now return cached result
        $cached = $this->securityService->getCachedValidationResult($smtpConfig);
        $this->assertEquals($result, $cached);
    }

    /** @test */
    public function it_does_not_cache_failed_validations()
    {
        $smtpConfig = [
            'host' => 'invalid.smtp.com',
            'port' => 587,
            'username' => 'test@invalid.com'
        ];

        // Cache a failed result
        $result = ['isValid' => false, 'message' => 'Failed'];
        $this->securityService->cacheValidationResult($smtpConfig, $result);

        // Should not cache failed results
        $cached = $this->securityService->getCachedValidationResult($smtpConfig);
        $this->assertNull($cached);
    }

    /** @test */
    public function it_gets_rate_limit_status()
    {
        $identifier = 'test_user_456';

        $status = $this->securityService->getRateLimitStatus($identifier);

        $this->assertArrayHasKey('minute', $status);
        $this->assertArrayHasKey('hour', $status);
        $this->assertArrayHasKey('attempts', $status['minute']);
        $this->assertArrayHasKey('max', $status['minute']);
        $this->assertArrayHasKey('remaining', $status['minute']);
    }
}