<?php

namespace Modules\Flowmaker\Models\Nodes;

// Include Google API Client
require_once base_path('google-api-client/vendor/autoload.php');

use Illuminate\Support\Facades\Log;
use Modules\Flowmaker\Models\Contact;

class Sheets extends Node
{
    // System service account credentials path
    private const SERVICE_ACCOUNT_PATH = 'test-sheet-node-project-93ba28a6ae9a.json';
    
    public function process($message, $data)
    {
        Log::info('🔄 Processing Sheets node', [
            'node_id' => $this->id ?? 'unknown',
            'flow_id' => $this->flow_id ?? 'unknown'
        ]);

        try {
            // Check if Google Client is available
            if (!class_exists('\Google\Client')) {
                Log::error('❌ Google Client class not found');
                return ['success' => false, 'error' => 'Google API Client not properly loaded'];
            }

            $nodeData = $this->getDataAsArray();
            $settings = $nodeData['settings'] ?? [];

            Log::debug('📋 Sheets settings', ['settings' => $settings]);

            // Resolve contact
            $contactId = is_object($data) ? ($data->contact_id ?? null) : ($data['contact_id'] ?? null);
            $contact = $contactId ? Contact::find($contactId) : null;
            
            if (!$contact) {
                Log::error('❌ Contact not found', ['contactId' => $contactId]);
                return ['success' => false, 'error' => 'Contact not found'];
            }

            // Extract and validate settings
            $operation = $settings['operation'] ?? 'read';
            $spreadsheetId = $settings['spreadsheetId'] ?? '';
            $range = $settings['range'] ?? 'A:Z';
            $authMethod = $settings['authMethod'] ?? 'service_account';
            $responseVar = trim($settings['responseVar'] ?? '');
            $dataToWrite = $settings['dataToWrite'] ?? '';
            $writeMode = $settings['writeMode'] ?? 'append';
            $sheetName = $settings['sheetName'] ?? 'New Sheet';

            // ✅ VALIDATE API KEY LIMITATIONS
            if ($authMethod === 'api_key' && in_array($operation, ['write', 'update', 'create_sheet'])) {
                Log::error('❌ API Key cannot perform write operations', [
                    'authMethod' => $authMethod,
                    'operation' => $operation
                ]);
                return ['success' => false, 'error' => 'API Key authentication only supports read operations. Please use Service Account for write operations.'];
            }

            // Process variables with contact data
            $spreadsheetId = $contact->changeVariables($spreadsheetId, $this->flow_id);
            $range = $contact->changeVariables($range, $this->flow_id);
            $dataToWrite = $contact->changeVariables($dataToWrite, $this->flow_id);

            // Extract spreadsheet ID from URL if provided
            if (preg_match('/\/spreadsheets\/d\/([a-zA-Z0-9-_]+)/', $spreadsheetId, $matches)) {
                $spreadsheetId = $matches[1];
            }

            if (empty($spreadsheetId)) {
                Log::error('❌ Spreadsheet ID is empty');
                return ['success' => false, 'error' => 'Spreadsheet ID is required'];
            }

            // Initialize Google Sheets client with proper authentication
            $client = $this->initializeGoogleClient();
            if (!$client) {
                return ['success' => false, 'error' => 'Failed to initialize Google Client'];
            }
            
            $service = new \Google\Service\Sheets($client);

            $result = null;

            // Execute operation
            switch ($operation) {
                case 'read':
                    $result = $this->readFromSheet($service, $spreadsheetId, $range);
                    break;
                case 'write':
                    $result = $this->writeToSheet($service, $spreadsheetId, $range, $dataToWrite, $writeMode);
                    break;
                case 'update':
                    $result = $this->updateSheet($service, $spreadsheetId, $range, $dataToWrite);
                    break;
                case 'create_sheet':
                    $result = $this->createSheet($service, $spreadsheetId, $sheetName);
                    break;
                default:
                    throw new \Exception("Unknown operation: {$operation}");
            }

            Log::info('✅ Sheets operation completed', [
                'operation' => $operation,
                'authMethod' => $authMethod,
                'spreadsheetId' => $spreadsheetId,
                'success' => true
            ]);

            // Store response in contact state
            if ($responseVar !== '') {
                $contact->setContactState($this->flow_id, $responseVar, json_encode($result));
                Log::info('✅ Stored response in contact state', ['variable' => $responseVar]);
            }

            // Continue to next node
            $nextNode = $this->getNextNodeId();
            if ($nextNode) {
                $nextNode->process($message, $data);
            }

            return ['success' => true, 'data' => $result];

        } catch (\Google\Service\Exception $e) {
            $error = json_decode($e->getMessage(), true);
            Log::error('❌ Google Sheets API error', ['error' => $error, 'trace' => $e->getTraceAsString()]);
            
            // Handle common errors with user-friendly messages
            if (isset($error['error']['code'])) {
                switch ($error['error']['code']) {
                    case 403:
                        return ['success' => false, 'error' => 'Access denied. Please share the spreadsheet with our service account or make it public for API key access.'];
                    case 404:
                        return ['success' => false, 'error' => 'Spreadsheet not found. Please verify the spreadsheet ID.'];
                    case 400:
                        return ['success' => false, 'error' => 'Invalid request. Please check your range format (e.g., A1:B10).'];
                    default:
                        return ['success' => false, 'error' => $error['error']['message'] ?? 'Google Sheets API error'];
                }
            }
            
            return ['success' => false, 'error' => $e->getMessage()];
            
        } catch (\Exception $e) {
            Log::error('❌ Sheets node error', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * ✅ FIXED: Initialize Google Client with proper authentication method
     */
    private function initializeGoogleClient()
    {
        try {
            // Get settings to check authMethod
            $nodeData = $this->getDataAsArray();
            $settings = $nodeData['settings'] ?? [];
            $authMethod = $settings['authMethod'] ?? 'service_account';
            
            Log::info('🔧 Initializing Google Client', ['authMethod' => $authMethod]);

            $client = new \Google\Client();
            $client->setApplicationName('Flowmaker Sheets Integration');
            $client->setScopes([\Google\Service\Sheets::SPREADSHEETS]);

            switch ($authMethod) {
                case 'service_account':
                    // Use system service account credentials
                    $credentialsPath = storage_path(self::SERVICE_ACCOUNT_PATH);
                    
                    Log::info('🔍 Looking for service account credentials at: ' . $credentialsPath);
                    
                    if (!file_exists($credentialsPath)) {
                        Log::error('❌ Service account credentials not found at: ' . $credentialsPath);
                        throw new \Exception('Service account credentials not found. Please upload the credentials file to storage/app/credentials/google-service-account.json');
                    }

                    $client->setAuthConfig($credentialsPath);
                    Log::info('✅ Using Service Account authentication');
                    break;

                case 'api_key':
                    // Use API key authentication
                    $apiKey = $settings['apiKey'] ?? '';
                    if (empty($apiKey)) {
                        Log::error('❌ API Key is empty');
                        throw new \Exception('API key is required for API key authentication');
                    }
                    
                    $client->setDeveloperKey($apiKey);
                    Log::info('✅ Using API Key authentication', ['apiKey' => substr($apiKey, 0, 10) . '...']);
                    break;

                default:
                    throw new \Exception("Unknown authentication method: {$authMethod}");
            }

            Log::info('✅ Google Client initialized successfully');
            return $client;
            
        } catch (\Exception $e) {
            Log::error('❌ Failed to initialize Google Client', ['error' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * Read data from Google Sheet
     */
    private function readFromSheet($service, $spreadsheetId, $range)
    {
        Log::info('📖 Reading from sheet', ['spreadsheetId' => $spreadsheetId, 'range' => $range]);
        
        $response = $service->spreadsheets_values->get($spreadsheetId, $range);
        $values = $response->getValues();
        
        return [
            'values' => $values ?? [],
            'range' => $range,
            'rowCount' => $values ? count($values) : 0
        ];
    }

    /**
     * Write data to Google Sheet
     */
    private function writeToSheet($service, $spreadsheetId, $range, $dataToWrite, $writeMode)
    {
        Log::info('✏️ Writing to sheet', ['spreadsheetId' => $spreadsheetId, 'range' => $range, 'mode' => $writeMode]);
        
        try {
            // Parse data (JSON, CSV, or simple text)
            $values = $this->parseDataToWrite($dataToWrite);
            
            $body = new \Google\Service\Sheets\ValueRange(['values' => $values]);
            
            $params = [
                'valueInputOption' => 'USER_ENTERED'
            ];

            if ($writeMode === 'append') {
                $response = $service->spreadsheets_values->append(
                    $spreadsheetId,
                    $range,
                    $body,
                    $params
                );
            
                // Append → first get the UpdateValuesResponse object
                $updates = $response->getUpdates();
            
                return [
                    'updatedRows'    => $updates ? $updates->getUpdatedRows()    : 0,
                    'updatedColumns' => $updates ? $updates->getUpdatedColumns() : 0,
                    'updatedCells'   => $updates ? $updates->getUpdatedCells()   : 0,
                    'writeMode'      => $writeMode,
                ];
            } else {
                $response = $service->spreadsheets_values->update(
                    $spreadsheetId,
                    $range,
                    $body,
                    $params
                );
            
                return [
                    'updatedRows'    => $response->getUpdatedRows()    ?? 0,
                    'updatedColumns' => $response->getUpdatedColumns() ?? 0,
                    'updatedCells'   => $response->getUpdatedCells()   ?? 0,
                    'writeMode'      => $writeMode,
                ];
            }
        } catch (\Exception $e) {
            Log::error('❌ Error writing to sheet', ['error' => $e->getMessage()]);
            throw $e;
        }
    }

    /**
     * Parse data to write into proper format
     */
    private function parseDataToWrite($dataToWrite)
    {
        // Try to parse as JSON first
        $decoded = json_decode($dataToWrite, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
            return $decoded;
        }

        // Try to parse as CSV
        if (strpos($dataToWrite, ',') !== false) {
            $lines = explode("\n", trim($dataToWrite));
            $result = [];
            foreach ($lines as $line) {
                if (trim($line)) {
                    $result[] = str_getcsv($line);
                }
            }
            return $result;
        }

        // Treat as simple text (single row)
        return [[$dataToWrite]];
    }

    /**
     * Update data in Google Sheet
     */
    private function updateSheet($service, $spreadsheetId, $range, $dataToWrite)
    {
        return $this->writeToSheet($service, $spreadsheetId, $range, $dataToWrite, 'update');
    }

    /**
     * Create new sheet in spreadsheet
     */
    private function createSheet($service, $spreadsheetId, $sheetName)
    {
        Log::info('📄 Creating new sheet', ['spreadsheetId' => $spreadsheetId, 'sheetName' => $sheetName]);
        
        $request = new \Google\Service\Sheets\AddSheetRequest();
        $request->setProperties(new \Google\Service\Sheets\SheetProperties());
        $request->getProperties()->setTitle($sheetName);

        $batchUpdateRequest = new \Google\Service\Sheets\BatchUpdateSpreadsheetRequest();
        $batchUpdateRequest->setRequests([$request]);

        $response = $service->spreadsheets->batchUpdate($spreadsheetId, $batchUpdateRequest);
        
        return [
            'sheetId' => $response->getReplies()[0]->getAddSheet()->getProperties()->getSheetId(),
            'sheetName' => $sheetName
        ];
    }

    /**
     * Get next node in the flow
     */
    protected function getNextNodeId($data = null)
    {
        if (!empty($this->outgoingEdges)) {
            return $this->outgoingEdges[0]->getTarget();
        }
        return null;
    }
}
