import React, { useEffect, useState } from 'react';
import { Handle, Position } from '@xyflow/react';
import { 
  Globe, 
  Trash2, 
  Upload, 
  X, 
  Plus, 
  Minus, 
  ArrowRight, 
  Search, 
  Settings, 
  FileText, 
  Code, 
  Database, 
  Route, 
  Layers,
  ChevronDown,
  ChevronRight,
  Zap,
  Clock,
  RefreshCw,
  Link,
  Filter,
  Send,
  Download,
  File,
  CheckCircle,
  AlertCircle,
  Info
} from 'lucide-react';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Button } from '@/components/ui/button';
import { useReactFlow } from '@xyflow/react';
import { useFlowActions } from '@/hooks/useFlowActions';
import { ContextMenu, ContextMenuContent, ContextMenuItem, ContextMenuTrigger } from '@/components/ui/context-menu';

interface NewRequestNodeProps {
  id: string;
  data: any;
}

type HTTPMethod = 'GET' | 'POST' | 'PUT' | 'PATCH' | 'DELETE' | 'HEAD' | 'OPTIONS';
type BodyType = 'json' | 'form' | 'xml' | 'multipart';
type ConditionType = 'status_code' | 'response_contains' | 'response_json';
type PaginationType = 'none' | 'page' | 'cursor' | 'offset';

interface FileUpload {
  id: string;
  name: string;
  file: File;
  fieldName: string;
}

interface Condition {
  id: string;
  type: ConditionType;
  operator: string;
  value: string;
  targetNodeId?: string;
}

interface PaginationConfig {
  type: PaginationType;
  pageParam: string;
  pageSizeParam: string;
  pageSize: number;
  maxPages: number;
  nextPagePath: string;
  hasMorePath: string;
  enableSendMore: boolean;
}

interface QueryParam {
  id: string;
  key: string;
  value: string;
}

interface NewRequestSettings {
  method: HTTPMethod;
  url: string;
  headers: string;
  body: string;
  bodyType: BodyType;
  responseVar: string;
  timeout: number;
  retry: number;
  files: FileUpload[];
  conditions: Condition[];
  enableConditionalRouting: boolean;
  pagination: PaginationConfig;
  queryParams: QueryParam[];
  enableQueryParams: boolean;
}

const defaultSettings: NewRequestSettings = {
  method: 'GET',
  url: '',
  headers: '{\n  "Content-Type": "application/json"\n}',
  body: '{\n  \n}',
  bodyType: 'json',
  responseVar: 'new_request_response',
  timeout: 10,
  retry: 0,
  files: [],
  conditions: [],
  enableConditionalRouting: false,
  pagination: {
    type: 'none',
    pageParam: 'page',
    pageSizeParam: 'per_page',
    pageSize: 10,
    maxPages: 5,
    nextPagePath: 'next_page_url',
    hasMorePath: 'has_more',
    enableSendMore: false,
  },
  queryParams: [],
  enableQueryParams: false,
};

const NewRequestNode = ({ id, data }: NewRequestNodeProps) => {
  const { setNodes } = useReactFlow();
  const { deleteNode } = useFlowActions();
  const [settings, setSettings] = useState<NewRequestSettings>({
    ...defaultSettings,
    ...(data.settings?.new_request || {})
  });
  const [headersError, setHeadersError] = useState<string | null>(null);
  const [bodyError, setBodyError] = useState<string | null>(null);
  const [expandedSections, setExpandedSections] = useState({
    basic: true,
    advanced: false,
    files: false,
    routing: false,
    pagination: false,
    queryParams: false
  });

  useEffect(() => {
    setNodes(nodes =>
      nodes.map(node => {
        if (node.id === id) {
          const currentData = (node.data || {}) as Record<string, unknown>;
          const currentSettings = ((currentData.settings || {}) as Record<string, unknown>);
          return {
            ...node,
            data: {
              ...currentData,
              settings: {
                ...currentSettings,
                new_request: settings
              }
            }
          };
        }
        return node;
      })
    );
  }, [settings, id, setNodes]);

  const updateSettings = (updates: Partial<NewRequestSettings>) => {
    setSettings(current => ({
      ...current,
      ...updates
    }));
  };

  const updatePagination = (updates: Partial<PaginationConfig>) => {
    updateSettings({
      pagination: { ...settings.pagination, ...updates }
    });
  };

  const toggleSection = (section: keyof typeof expandedSections) => {
    setExpandedSections(prev => ({
      ...prev,
      [section]: !prev[section]
    }));
  };

  // Validate JSON for headers and body (only for JSON body type)
  useEffect(() => {
    try {
      JSON.parse(settings.headers || '{}');
      setHeadersError(null);
    } catch (e) {
      setHeadersError('Invalid JSON');
    }
    
    if (settings.bodyType === 'json') {
      try {
        JSON.parse(settings.body || '{}');
        setBodyError(null);
      } catch (e) {
        setBodyError('Invalid JSON');
      }
    } else {
      setBodyError(null);
    }
  }, [settings.headers, settings.body, settings.bodyType]);

  const getBodyPlaceholder = () => {
    switch (settings.bodyType) {
      case 'json':
        return '{\n  "key": "value"\n}';
      case 'form':
        return 'key1=value1\nkey2=value2';
      case 'xml':
        return '<root>\n  <element>value</element>\n</root>';
      case 'multipart':
        return 'Form data will be handled automatically with files';
      default:
        return '';
    }
  };

  const getBodyLabel = () => {
    switch (settings.bodyType) {
      case 'json':
        return 'Body (JSON)';
      case 'form':
        return 'Body (Form Data)';
      case 'xml':
        return 'Body (XML)';
      case 'multipart':
        return 'Body (Multipart)';
      default:
        return 'Body';
    }
  };

  const getMethodColor = (method: HTTPMethod) => {
    switch (method) {
      case 'GET': return 'bg-green-100 text-green-800 border-green-200';
      case 'POST': return 'bg-blue-100 text-blue-800 border-blue-200';
      case 'PUT': return 'bg-yellow-100 text-yellow-800 border-yellow-200';
      case 'PATCH': return 'bg-orange-100 text-orange-800 border-orange-200';
      case 'DELETE': return 'bg-red-100 text-red-800 border-red-200';
      case 'HEAD': return 'bg-purple-100 text-purple-800 border-purple-200';
      case 'OPTIONS': return 'bg-gray-100 text-gray-800 border-gray-200';
      default: return 'bg-gray-100 text-gray-800 border-gray-200';
    }
  };

  const getBodyTypeIcon = (type: BodyType) => {
    switch (type) {
      case 'json': return <Code className="h-4 w-4" />;
      case 'form': return <Database className="h-4 w-4" />;
      case 'xml': return <FileText className="h-4 w-4" />;
      case 'multipart': return <Upload className="h-4 w-4" />;
      default: return <FileText className="h-4 w-4" />;
    }
  };

  const handleFileUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
    const files = event.target.files;
    if (!files) return;

    const newFiles: FileUpload[] = Array.from(files).map(file => ({
      id: Math.random().toString(36).substr(2, 9),
      name: file.name,
      file: file,
      fieldName: file.name.split('.')[0] || 'file'
    }));

    updateSettings({
      files: [...settings.files, ...newFiles]
    });
  };

  const removeFile = (fileId: string) => {
    updateSettings({
      files: settings.files.filter(f => f.id !== fileId)
    });
  };

  const updateFileFieldName = (fileId: string, fieldName: string) => {
    updateSettings({
      files: settings.files.map(f => 
        f.id === fileId ? { ...f, fieldName } : f
      )
    });
  };

  const addCondition = () => {
    const newCondition: Condition = {
      id: Math.random().toString(36).substr(2, 9),
      type: 'status_code',
      operator: 'equals',
      value: '200',
    };
    updateSettings({
      conditions: [...settings.conditions, newCondition]
    });
  };

  const removeCondition = (conditionId: string) => {
    updateSettings({
      conditions: settings.conditions.filter(c => c.id !== conditionId)
    });
  };

  const updateCondition = (conditionId: string, updates: Partial<Condition>) => {
    updateSettings({
      conditions: settings.conditions.map(c => 
        c.id === conditionId ? { ...c, ...updates } : c
      )
    });
  };

  const addQueryParam = () => {
    const newParam: QueryParam = {
      id: Math.random().toString(36).substr(2, 9),
      key: '',
      value: '',
    };
    updateSettings({
      queryParams: [...settings.queryParams, newParam]
    });
  };

  const removeQueryParam = (paramId: string) => {
    updateSettings({
      queryParams: settings.queryParams.filter(p => p.id !== paramId)
    });
  };

  const updateQueryParam = (paramId: string, updates: Partial<QueryParam>) => {
    updateSettings({
      queryParams: settings.queryParams.map(p => 
        p.id === paramId ? { ...p, ...updates } : p
      )
    });
  };

  const getOperators = (type: ConditionType) => {
    switch (type) {
      case 'status_code':
        return [
          { value: 'equals', label: 'Equals' },
          { value: 'not_equals', label: 'Not Equals' },
          { value: 'greater_than', label: 'Greater Than' },
          { value: 'less_than', label: 'Less Than' },
          { value: 'between', label: 'Between' }
        ];
      case 'response_contains':
        return [
          { value: 'contains', label: 'Contains' },
          { value: 'not_contains', label: 'Not Contains' },
          { value: 'starts_with', label: 'Starts With' },
          { value: 'ends_with', label: 'Ends With' }
        ];
      case 'response_json':
        return [
          { value: 'equals', label: 'Equals' },
          { value: 'not_equals', label: 'Not Equals' },
          { value: 'exists', label: 'Exists' },
          { value: 'not_exists', label: 'Not Exists' }
        ];
      default:
        return [];
    }
  };

  return (
    <ContextMenu>
      <ContextMenuTrigger>
        <div className="w-[600px] bg-white rounded-xl shadow-lg border border-gray-200">
          <Handle
            type="target"
            position={Position.Left}
            className="!bg-blue-500 !w-4 !h-4 !rounded-full"
          />

          {/* Header */}
          <div className="flex items-center gap-3 p-4 bg-gradient-to-r from-blue-50 to-indigo-50 border-b border-gray-200 rounded-t-xl">
            <div className="p-2 bg-blue-100 rounded-lg">
              <Globe className="h-5 w-5 text-blue-600" />
            </div>
            <div className="flex-1">
              <div className="font-semibold text-gray-900">Advanced HTTP Request</div>
              <div className="text-sm text-gray-600">Make powerful API calls with advanced features</div>
            </div>
            <div className={`px-3 py-1 rounded-full text-xs font-medium border ${getMethodColor(settings.method)}`}>
              {settings.method}
            </div>
          </div>

          <div className="p-4 space-y-4">
            {/* Basic Settings Section */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div 
                className="flex items-center gap-2 cursor-pointer mb-3"
                onClick={() => toggleSection('basic')}
              >
                {expandedSections.basic ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                <Settings className="h-4 w-4 text-gray-600" />
                <span className="font-medium text-gray-900">Basic Settings</span>
              </div>
              
              {expandedSections.basic && (
                <div className="space-y-4">
                  <div className="grid grid-cols-3 gap-3">
                    <div>
                      <Label className="text-sm font-medium text-gray-700">Method</Label>
                      <Select
                        value={settings.method}
                        onValueChange={(value: HTTPMethod) => updateSettings({ method: value })}
                      >
                        <SelectTrigger className="mt-1">
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="GET">GET</SelectItem>
                          <SelectItem value="POST">POST</SelectItem>
                          <SelectItem value="PUT">PUT</SelectItem>
                          <SelectItem value="PATCH">PATCH</SelectItem>
                          <SelectItem value="DELETE">DELETE</SelectItem>
                          <SelectItem value="HEAD">HEAD</SelectItem>
                          <SelectItem value="OPTIONS">OPTIONS</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                    <div className="col-span-2">
                      <Label className="text-sm font-medium text-gray-700">Response Variable</Label>
                      <Input
                        placeholder="e.g., api_response"
                        value={settings.responseVar}
                        onChange={e => updateSettings({ responseVar: e.target.value })}
                        className="mt-1"
                      />
                    </div>
                  </div>

                  <div>
                    <Label className="text-sm font-medium text-gray-700">Request URL</Label>
                    <div className="mt-1 relative">
                      <Link className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
                      <Input
                        placeholder="https://api.example.com/endpoint"
                        value={settings.url}
                        onChange={e => updateSettings({ url: e.target.value })}
                        className="pl-10"
                      />
                    </div>
                  </div>

                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label className="text-sm font-medium text-gray-700 flex items-center gap-2">
                        <Clock className="h-4 w-4" />
                        Timeout (seconds)
                      </Label>
                      <Input
                        type="number"
                        min={1}
                        max={300}
                        value={settings.timeout || 10}
                        onChange={e => updateSettings({ timeout: Number(e.target.value) })}
                        className="mt-1"
                      />
                    </div>
                    <div>
                      <Label className="text-sm font-medium text-gray-700 flex items-center gap-2">
                        <RefreshCw className="h-4 w-4" />
                        Retry Count
                      </Label>
                      <Input
                        type="number"
                        min={0}
                        max={10}
                        value={settings.retry || 0}
                        onChange={e => updateSettings({ retry: Number(e.target.value) })}
                        className="mt-1"
                      />
                    </div>
                  </div>
                </div>
              )}
            </div>

            {/* Advanced Settings Section */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div 
                className="flex items-center gap-2 cursor-pointer mb-3"
                onClick={() => toggleSection('advanced')}
              >
                {expandedSections.advanced ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                <Zap className="h-4 w-4 text-gray-600" />
                <span className="font-medium text-gray-900">Advanced Settings</span>
              </div>
              
              {expandedSections.advanced && (
                <div className="space-y-4">
                  <div>
                    <Label className="text-sm font-medium text-gray-700">Headers (JSON)</Label>
                    <textarea
                      className="w-full border rounded-lg p-3 text-sm font-mono mt-1 bg-white"
                      rows={3}
                      placeholder={`{
  "Authorization": "Bearer ...",
  "Content-Type": "application/json"
}`}
                      value={settings.headers}
                      onChange={e => updateSettings({ headers: e.target.value })}
                    />
                    {headersError && (
                      <div className="flex items-center gap-1 text-red-500 text-xs mt-1">
                        <AlertCircle className="h-3 w-3" />
                        {headersError}
                      </div>
                    )}
                  </div>

                  <div>
                    <Label className="text-sm font-medium text-gray-700 flex items-center gap-2">
                      {getBodyTypeIcon(settings.bodyType)}
                      Body Type
                    </Label>
                    <Select
                      value={settings.bodyType}
                      onValueChange={(value: BodyType) => updateSettings({ bodyType: value })}
                    >
                      <SelectTrigger className="mt-1">
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="json">JSON</SelectItem>
                        <SelectItem value="form">Form Data</SelectItem>
                        <SelectItem value="xml">XML</SelectItem>
                        <SelectItem value="multipart">Multipart</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>

                  <div>
                    <Label className="text-sm font-medium text-gray-700">{getBodyLabel()}</Label>
                    <textarea
                      className="w-full border rounded-lg p-3 text-sm font-mono mt-1 bg-white"
                      rows={5}
                      placeholder={getBodyPlaceholder()}
                      value={settings.body}
                      onChange={e => updateSettings({ body: e.target.value })}
                      disabled={settings.bodyType === 'multipart'}
                    />
                    {bodyError && (
                      <div className="flex items-center gap-1 text-red-500 text-xs mt-1">
                        <AlertCircle className="h-3 w-3" />
                        {bodyError}
                      </div>
                    )}
                  </div>
                </div>
              )}
            </div>

            {/* Query Parameters Section */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div 
                className="flex items-center gap-2 cursor-pointer mb-3"
                onClick={() => toggleSection('queryParams')}
              >
                {expandedSections.queryParams ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                <Search className="h-4 w-4 text-gray-600" />
                <span className="font-medium text-gray-900">Query Parameters</span>
                {settings.queryParams.length > 0 && (
                  <span className="ml-2 px-2 py-1 bg-blue-100 text-blue-800 text-xs rounded-full">
                    {settings.queryParams.length}
                  </span>
                )}
              </div>
              
              {expandedSections.queryParams && (
                <div className="space-y-3">
                  {settings.queryParams.map((param) => (
                    <div key={param.id} className="flex items-center gap-2 bg-white p-3 rounded-lg border">
                      <Input
                        placeholder="Parameter name"
                        value={param.key}
                        onChange={(e) => updateQueryParam(param.id, { key: e.target.value })}
                        className="flex-1"
                      />
                      <Input
                        placeholder="Value"
                        value={param.value}
                        onChange={(e) => updateQueryParam(param.id, { value: e.target.value })}
                        className="flex-1"
                      />
                      <Button
                        type="button"
                        variant="ghost"
                        size="sm"
                        onClick={() => removeQueryParam(param.id)}
                        className="text-red-500 hover:text-red-700"
                      >
                        <X className="h-4 w-4" />
                      </Button>
                    </div>
                  ))}

                  <Button
                    type="button"
                    variant="outline"
                    size="sm"
                    onClick={addQueryParam}
                    className="w-full"
                  >
                    <Plus className="h-4 w-4 mr-2" />
                    Add Query Parameter
                  </Button>
                </div>
              )}
            </div>

            {/* File Upload Section */}
            {settings.bodyType === 'multipart' && (
              <div className="bg-gray-50 rounded-lg p-4">
                <div 
                  className="flex items-center gap-2 cursor-pointer mb-3"
                  onClick={() => toggleSection('files')}
                >
                  {expandedSections.files ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                  <File className="h-4 w-4 text-gray-600" />
                  <span className="font-medium text-gray-900">File Uploads</span>
                  {settings.files.length > 0 && (
                    <span className="ml-2 px-2 py-1 bg-green-100 text-green-800 text-xs rounded-full">
                      {settings.files.length}
                    </span>
                  )}
                </div>
                
                {expandedSections.files && (
                  <div className="space-y-3">
                    <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 bg-white">
                      <div className="text-center">
                        <Upload className="mx-auto h-8 w-8 text-gray-400 mb-2" />
                        <p className="text-sm text-gray-600 mb-2">
                          Click to upload files or drag and drop
                        </p>
                        <Button
                          type="button"
                          variant="outline"
                          onClick={() => document.getElementById(`file-upload-${id}`)?.click()}
                        >
                          Choose Files
                        </Button>
                        <input
                          id={`file-upload-${id}`}
                          type="file"
                          multiple
                          className="hidden"
                          onChange={handleFileUpload}
                        />
                      </div>
                    </div>
                    
                    {settings.files.length > 0 && (
                      <div className="space-y-2">
                        {settings.files.map((file) => (
                          <div key={file.id} className="flex items-center gap-3 p-3 bg-white rounded-lg border">
                            <File className="h-5 w-5 text-blue-500" />
                            <div className="flex-1">
                              <div className="text-sm font-medium text-gray-900">{file.name}</div>
                              <Input
                                type="text"
                                placeholder="Field name"
                                value={file.fieldName}
                                onChange={(e) => updateFileFieldName(file.id, e.target.value)}
                                className="text-xs mt-1"
                              />
                            </div>
                            <Button
                              type="button"
                              variant="ghost"
                              size="sm"
                              onClick={() => removeFile(file.id)}
                              className="text-red-500 hover:text-red-700"
                            >
                              <X className="h-4 w-4" />
                            </Button>
                          </div>
                        ))}
                      </div>
                    )}
                  </div>
                )}
              </div>
            )}

            {/* Conditional Routing Section */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div 
                className="flex items-center gap-2 cursor-pointer mb-3"
                onClick={() => toggleSection('routing')}
              >
                {expandedSections.routing ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                <Route className="h-4 w-4 text-gray-600" />
                <span className="font-medium text-gray-900">Conditional Routing</span>
                {settings.conditions.length > 0 && (
                  <span className="ml-2 px-2 py-1 bg-purple-100 text-purple-800 text-xs rounded-full">
                    {settings.conditions.length}
                  </span>
                )}
              </div>
              
              {expandedSections.routing && (
                <div className="space-y-3">
                  {settings.conditions.map((condition) => (
                    <div key={condition.id} className="bg-white p-4 rounded-lg border">
                      <div className="flex items-center gap-2 mb-3">
                        <Filter className="h-4 w-4 text-purple-500" />
                        <span className="text-sm font-medium text-gray-900">Condition</span>
                      </div>
                      <div className="grid grid-cols-3 gap-2">
                        <Select
                          value={condition.type}
                          onValueChange={(value: ConditionType) => 
                            updateCondition(condition.id, { type: value, operator: getOperators(value)[0]?.value || 'equals' })
                          }
                        >
                          <SelectTrigger>
                            <SelectValue />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="status_code">Status Code</SelectItem>
                            <SelectItem value="response_contains">Response Contains</SelectItem>
                            <SelectItem value="response_json">Response JSON</SelectItem>
                          </SelectContent>
                        </Select>

                        <Select
                          value={condition.operator}
                          onValueChange={(value) => updateCondition(condition.id, { operator: value })}
                        >
                          <SelectTrigger>
                            <SelectValue />
                          </SelectTrigger>
                          <SelectContent>
                            {getOperators(condition.type).map(op => (
                              <SelectItem key={op.value} value={op.value}>
                                {op.label}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>

                        <div className="flex gap-2">
                          <Input
                            placeholder="Value"
                            value={condition.value}
                            onChange={(e) => updateCondition(condition.id, { value: e.target.value })}
                            className="flex-1"
                          />
                          <Button
                            type="button"
                            variant="ghost"
                            size="sm"
                            onClick={() => removeCondition(condition.id)}
                            className="text-red-500 hover:text-red-700"
                          >
                            <X className="h-4 w-4" />
                          </Button>
                        </div>
                      </div>
                    </div>
                  ))}

                  <Button
                    type="button"
                    variant="outline"
                    size="sm"
                    onClick={addCondition}
                    className="w-full"
                  >
                    <Plus className="h-4 w-4 mr-2" />
                    Add Condition
                  </Button>
                </div>
              )}
            </div>

            {/* Pagination Section */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div 
                className="flex items-center gap-2 cursor-pointer mb-3"
                onClick={() => toggleSection('pagination')}
              >
                {expandedSections.pagination ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                <Layers className="h-4 w-4 text-gray-600" />
                <span className="font-medium text-gray-900">Pagination & Send More</span>
                {settings.pagination.enableSendMore && (
                  <span className="ml-2 px-2 py-1 bg-orange-100 text-orange-800 text-xs rounded-full">
                    Active
                  </span>
                )}
              </div>
              
              {expandedSections.pagination && (
                <div className="space-y-4">
                  <div className="bg-white p-4 rounded-lg border">
                    <div className="flex items-center gap-3 mb-4">
                      <div className="flex items-center gap-2">
                        <input
                          type="checkbox"
                          id={`pagination-${id}`}
                          checked={settings.pagination.enableSendMore}
                          onChange={(e) => updatePagination({ enableSendMore: e.target.checked })}
                          className="rounded"
                        />
                        <Label htmlFor={`pagination-${id}`} className="text-sm font-medium text-gray-700">
                          Enable Pagination / Send More
                        </Label>
                      </div>
                      {settings.pagination.enableSendMore && (
                        <div className="flex items-center gap-2 px-2 py-1 bg-green-100 text-green-800 text-xs rounded-full">
                          <CheckCircle className="h-3 w-3" />
                          Enabled
                        </div>
                      )}
                    </div>

                    {settings.pagination.enableSendMore && (
                      <div className="space-y-4">
                        <div>
                          <Label className="text-sm font-medium text-gray-700 flex items-center gap-2">
                            <Database className="h-4 w-4" />
                            Pagination Type
                          </Label>
                          <Select
                            value={settings.pagination.type}
                            onValueChange={(value: PaginationType) => updatePagination({ type: value })}
                          >
                            <SelectTrigger className="mt-1">
                              <SelectValue />
                            </SelectTrigger>
                            <SelectContent>
                              <SelectItem value="none">None</SelectItem>
                              <SelectItem value="page">Page Number</SelectItem>
                              <SelectItem value="cursor">Cursor-based</SelectItem>
                              <SelectItem value="offset">Offset-based</SelectItem>
                            </SelectContent>
                          </Select>
                        </div>

                        {settings.pagination.type !== 'none' && (
                          <div className="space-y-4">
                            <div className="grid grid-cols-2 gap-3">
                              <div>
                                <Label className="text-sm font-medium text-gray-700">Page Parameter</Label>
                                <Input
                                  placeholder="page"
                                  value={settings.pagination.pageParam}
                                  onChange={(e) => updatePagination({ pageParam: e.target.value })}
                                  className="mt-1"
                                />
                              </div>
                              <div>
                                <Label className="text-sm font-medium text-gray-700">Page Size Parameter</Label>
                                <Input
                                  placeholder="per_page"
                                  value={settings.pagination.pageSizeParam}
                                  onChange={(e) => updatePagination({ pageSizeParam: e.target.value })}
                                  className="mt-1"
                                />
                              </div>
                            </div>

                            <div className="grid grid-cols-2 gap-3">
                              <div>
                                <Label className="text-sm font-medium text-gray-700">Page Size</Label>
                                <Input
                                  type="number"
                                  min={1}
                                  max={100}
                                  value={settings.pagination.pageSize}
                                  onChange={(e) => updatePagination({ pageSize: Number(e.target.value) })}
                                  className="mt-1"
                                />
                              </div>
                              <div>
                                <Label className="text-sm font-medium text-gray-700">Max Pages</Label>
                                <Input
                                  type="number"
                                  min={1}
                                  max={20}
                                  value={settings.pagination.maxPages}
                                  onChange={(e) => updatePagination({ maxPages: Number(e.target.value) })}
                                  className="mt-1"
                                />
                              </div>
                            </div>

                            <div>
                              <Label className="text-sm font-medium text-gray-700 flex items-center gap-2">
                                <ArrowRight className="h-4 w-4" />
                                Next Page Path (JSON path)
                              </Label>
                              <Input
                                placeholder="next_page_url"
                                value={settings.pagination.nextPagePath}
                                onChange={(e) => updatePagination({ nextPagePath: e.target.value })}
                                className="mt-1"
                              />
                            </div>

                            <div>
                              <Label className="text-sm font-medium text-gray-700 flex items-center gap-2">
                                <Info className="h-4 w-4" />
                                Has More Path (JSON path)
                              </Label>
                              <Input
                                placeholder="has_more"
                                value={settings.pagination.hasMorePath}
                                onChange={(e) => updatePagination({ hasMorePath: e.target.value })}
                                className="mt-1"
                              />
                            </div>
                          </div>
                        )}
                      </div>
                    )}
                  </div>
                </div>
              )}
            </div>
          </div>

          <Handle
            type="source"
            position={Position.Right}
            className="!bg-green-500 !w-4 !h-4 !rounded-full"
          />
        </div>
      </ContextMenuTrigger>
      <ContextMenuContent>
        <ContextMenuItem
          className="text-red-600 focus:text-red-600 focus:bg-red-100"
          onClick={() => deleteNode(id)}
        >
          <Trash2 className="mr-2 h-4 w-4" />
          Delete Node
        </ContextMenuItem>
      </ContextMenuContent>
    </ContextMenu>
  );
};

export default NewRequestNode; 