# CRM Integration for Contacts Module

## Overview

This implementation provides a complete CRM integration system for the Laravel Contacts module, enabling seamless import and synchronization of contacts from external CRM systems like HubSpot, Salesforce, and Pipedrive.

## Features

### ✅ Core Features
- **Plugin-based Architecture**: Easy to add new CRM connectors
- **Multi-tenancy Support**: Company-scoped data isolation
- **Secure Credential Storage**: Encrypted credentials using Laravel's casting
- **Real-time Import Progress**: Live progress tracking with WebSocket-like updates
- **Duplicate Detection**: Smart duplicate detection with multiple strategies
- **Field Mapping**: Flexible field mapping between CRM and local contacts
- **Import History**: Complete audit trail of all import operations
- **Connection Health Monitoring**: Real-time connection status monitoring
- **Background Processing**: Support for large imports with progress tracking

### 🔧 Supported CRMs
- **HubSpot** (Fully implemented)
- **Salesforce** (Architecture ready)
- **Pipedrive** (Architecture ready)

## Architecture

### Database Schema

#### 1. `crm_connections` Table
Stores CRM connection configurations with encrypted credentials.

```sql
- id (Primary Key)
- company_id (Foreign Key to companies)
- name (Connection name)
- type (hubspot, salesforce, pipedrive)
- credentials (Encrypted JSON)
- field_mapping (JSON configuration)
- is_active (Boolean)
- is_healthy (Boolean)
- last_sync_at (Timestamp)
- last_test_at (Timestamp)
- error_message (Text)
- timestamps, soft_deletes
```

#### 2. `contact_crm_mapping` Table
Maps local contacts to CRM contacts.

```sql
- id (Primary Key)
- company_id (Foreign Key to companies)
- contact_id (Foreign Key to contacts)
- crm_connection_id (Foreign Key to crm_connections)
- crm_contact_id (External CRM contact ID)
- sync_status (synced, pending, failed, conflict)
- crm_data (JSON - stores CRM contact data)
- last_synced_at (Timestamp)
- sync_error (Text)
- timestamps, soft_deletes
```

#### 3. `crm_sync_history` Table
Tracks all import/sync operations.

```sql
- id (Primary Key)
- company_id (Foreign Key to companies)
- crm_connection_id (Foreign Key to crm_connections)
- sync_type (import, export, sync)
- status (pending, in_progress, completed, failed, cancelled)
- total_records (Integer)
- processed_records (Integer)
- successful_records (Integer)
- failed_records (Integer)
- duplicate_records (Integer)
- import_settings (JSON)
- field_mapping (JSON)
- error_message (Text)
- started_at (Timestamp)
- completed_at (Timestamp)
- timestamps, soft_deletes
```

### Service Architecture

#### 1. CRM Connector Interface
```php
interface CrmConnectorInterface
{
    public function testConnection(array $credentials): bool;
    public function getContacts(array $credentials, array $options = []): array;
    public function getContact(array $credentials, string $contactId): ?array;
    public function createContact(array $credentials, array $contactData): array;
    public function updateContact(array $credentials, string $contactId, array $contactData): array;
    public function deleteContact(array $credentials, string $contactId): bool;
    public function getAvailableFields(array $credentials): array;
    public function getType(): string;
    public function getDisplayName(): string;
    public function getRequiredCredentials(): array;
    public function validateCredentials(array $credentials): bool;
}
```

#### 2. Plugin Registry
Manages available CRM connectors and provides a unified interface.

#### 3. Enhanced CRM Import Service
Handles the complete import process with duplicate detection and progress tracking.

#### 4. Duplicate Detection Service
Implements smart duplicate detection using phone normalization and name similarity.

#### 5. Phone Normalization Service
Normalizes phone numbers for consistent duplicate detection.

## Installation

### 1. Run Migrations
```bash
php artisan migrate
```

### 2. Clear Cache
```bash
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

### 3. Publish Assets (if needed)
```bash
php artisan vendor:publish --tag=contacts-config
```

## Usage

### 1. Creating a CRM Connection

1. Navigate to **Contacts > CRM Connections**
2. Click **"Add Connection"**
3. Select CRM type (HubSpot, Salesforce, Pipedrive)
4. Enter connection name and credentials
5. Test the connection
6. Configure field mapping
7. Save the connection

### 2. Importing Contacts

#### Method 1: Quick Import
1. Go to **Contacts > Import**
2. Switch to **"CRM Import"** tab
3. Select a connection
4. Choose duplicate strategy
5. Click **"Import from CRM"**

#### Method 2: Advanced Import
1. Go to **Contacts > CRM Import**
2. Configure detailed import settings
3. Set batch size and other options
4. Start the import process

### 3. Monitoring Imports

1. View real-time progress in the import modal
2. Check **Import History** for completed imports
3. Monitor connection health in **CRM Connections**

## Configuration

### CRM Settings
Located in `modules/Contacts/Config/config.php`:

```php
'crm' => [
    'enabled' => true,
    'supported_types' => [
        'hubspot' => [
            'name' => 'HubSpot',
            'enabled' => true,
            'api_base_url' => 'https://api.hubapi.com',
            'rate_limit' => [
                'requests_per_minute' => 100,
                'burst_limit' => 10,
            ],
        ],
        // ... other CRMs
    ],
    'import' => [
        'default_batch_size' => 100,
        'max_batch_size' => 1000,
        'progress_update_interval' => 2,
        'timeout' => 300,
    ],
    // ... other settings
],
```

## API Endpoints

### CRM Connections
- `GET /contacts/crm` - List connections
- `POST /contacts/crm` - Create connection
- `GET /contacts/crm/{id}` - Show connection
- `PUT /contacts/crm/{id}` - Update connection
- `DELETE /contacts/crm/{id}` - Delete connection
- `POST /contacts/crm/test` - Test connection
- `POST /contacts/crm/fields` - Get CRM fields
- `POST /contacts/crm/{id}/toggle-status` - Toggle status

### CRM Import
- `GET /contacts/crm-import` - Import interface
- `POST /contacts/crm-import` - Start import
- `GET /contacts/crm-import/progress` - Get progress
- `POST /contacts/crm-import/cancel` - Cancel import

## Security Features

### 1. Encrypted Credentials
All CRM credentials are encrypted using Laravel's encrypted casting.

### 2. Company Isolation
All data is scoped to the current company using CompanyScope.

### 3. Input Validation
Comprehensive validation using Form Request classes.

### 4. Rate Limiting
Built-in rate limiting for API calls to prevent abuse.

### 5. CSRF Protection
All forms include CSRF protection.

## Error Handling

### 1. Connection Errors
- Invalid credentials
- Network timeouts
- API rate limits
- Authentication failures

### 2. Import Errors
- Duplicate detection conflicts
- Field mapping errors
- Data validation failures
- Processing timeouts

### 3. Recovery Mechanisms
- Automatic retry with exponential backoff
- Graceful error handling with user feedback
- Detailed error logging for debugging

## Performance Considerations

### 1. Batch Processing
- Configurable batch sizes (1-1000 records)
- Memory-efficient processing
- Progress tracking for large imports

### 2. Caching
- Connection health caching
- Field mapping caching
- Progress data caching

### 3. Database Optimization
- Proper indexing on foreign keys
- Soft deletes for data retention
- JSON columns for flexible data storage

## Testing

### 1. Unit Tests
```bash
php artisan test --filter=CrmIntegration
```

### 2. Feature Tests
```bash
php artisan test --filter=CrmConnectionTest
```

### 3. Browser Tests
```bash
php artisan dusk --filter=CrmImportTest
```

## Troubleshooting

### Common Issues

#### 1. Connection Test Fails
- Verify credentials are correct
- Check network connectivity
- Ensure CRM API is accessible
- Verify rate limits

#### 2. Import Stuck
- Check for PHP timeout settings
- Verify memory limits
- Check database connections
- Review error logs

#### 3. Duplicate Detection Issues
- Verify phone number formats
- Check field mapping configuration
- Review duplicate strategy settings

### Debug Mode
Enable detailed logging in `config/contacts.php`:

```php
'logging' => [
    'enabled' => true,
    'log_level' => 'debug',
    'log_imports' => true,
    'log_errors' => true,
    'log_api_calls' => true,
],
```

## Extending the System

### Adding New CRM Connectors

1. Create a new connector class:
```php
class NewCrmConnector implements CrmConnectorInterface
{
    // Implement all required methods
}
```

2. Register in `CrmPluginRegistry`:
```php
$this->register('newcrm', new NewCrmConnector());
```

3. Add configuration in `config.php`:
```php
'newcrm' => [
    'name' => 'New CRM',
    'enabled' => true,
    // ... other settings
],
```

### Custom Field Mappings

Override default mappings in the configuration:

```php
'field_mapping' => [
    'default_mappings' => [
        'name' => 'custom_name_field',
        'email' => 'custom_email_field',
        // ... other mappings
    ],
],
```

## Support

For issues and questions:
1. Check the troubleshooting section
2. Review error logs
3. Test with a simple connection first
4. Verify all dependencies are installed

## Changelog

### Version 1.0.0
- Initial CRM integration implementation
- HubSpot connector
- Duplicate detection system
- Real-time import progress
- Complete UI implementation
- Security and validation
- Comprehensive documentation

## License

This implementation follows the same license as the main Laravel application.