<?php

namespace Modules\Contacts\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Modules\Contacts\Models\CrmConnection;

class CrmImportRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return auth()->check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return [
            'crm_connection_id' => 'required|integer|exists:crm_connections,id',
            'duplicate_strategy' => 'required|string|in:skip,update,merge',
            'batch_size' => 'integer|min:1|max:1000',
            'contact_type' => 'nullable|string|in:individual,company,both',
            'properties' => 'array',
            'properties.*' => 'string',
            'assign_to_group' => 'integer|exists:groups,id|nullable',
            'import_settings' => 'array',
            'import_settings.auto_assign_country' => 'boolean',
            'import_settings.create_missing_groups' => 'boolean',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array
     */
    public function messages()
    {
        return [
            'crm_connection_id.required' => __('crm.validation.connection_required'),
            'crm_connection_id.exists' => __('crm.validation.connection_exists'),
            'duplicate_strategy.required' => __('crm.validation.duplicate_strategy_required'),
            'duplicate_strategy.in' => __('crm.validation.duplicate_strategy_invalid'),
            'batch_size.integer' => __('crm.validation.batch_size_integer'),
            'batch_size.min' => __('crm.validation.batch_size_min'),
            'batch_size.max' => __('crm.validation.batch_size_max'),
            'properties.array' => __('crm.validation.properties_array'),
            'assign_to_group.exists' => __('crm.validation.group_exists'),
            'import_settings.array' => __('crm.validation.import_settings_array'),
        ];
    }

    /**
     * Get custom attributes for validator errors.
     *
     * @return array
     */
    public function attributes()
    {
        return [
            'crm_connection_id' => __('crm.attributes.connection'),
            'duplicate_strategy' => __('crm.attributes.duplicate_strategy'),
            'batch_size' => __('crm.attributes.batch_size'),
            'properties' => __('crm.attributes.properties'),
            'assign_to_group' => __('crm.attributes.assign_to_group'),
            'import_settings' => __('crm.attributes.import_settings'),
        ];
    }

    /**
     * Configure the validator instance.
     *
     * @param  \Illuminate\Validation\Validator  $validator
     * @return void
     */
    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            // Validate that the CRM connection is active and healthy
            if ($this->has('crm_connection_id')) {
                $connection = CrmConnection::find($this->crm_connection_id);
                
                if ($connection && !$connection->isHealthy()) {
                    $validator->errors()->add('crm_connection_id', __('crm.validation.connection_unhealthy'));
                }
            }
        });
    }

    /**
     * Get the validated data with defaults.
     *
     * @param string|null $key
     * @param mixed $default
     * @return array
     */
    public function validated($key = null, $default = null)
    {
        $validated = parent::validated($key, $default);
        
        // If a specific key is requested, return it directly
        if ($key !== null) {
            return $validated;
        }
        
        // Set defaults for the full validated data
        $validated['batch_size'] = min($validated['batch_size'] ?? 100, 100); // HubSpot limit is 100
        $validated['properties'] = $validated['properties'] ?? ['firstname', 'lastname', 'email', 'phone', 'company'];
        $validated['import_settings'] = array_merge([
            'auto_assign_country' => true,
            'create_missing_groups' => false,
        ], $validated['import_settings'] ?? []);

        return $validated;
    }
}