<?php

namespace Modules\Contacts\Imports;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Modules\Contacts\Models\Contact;
use Modules\Contacts\Models\Field;
use Modules\Contacts\Services\PhoneNormalizationService;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Illuminate\Support\Facades\Log;

class ContactsImport implements ToModel, WithHeadingRow, WithChunkReading
{
    private $phoneService;
    private $importStats;

    public function __construct()
    {
        $this->phoneService = new PhoneNormalizationService();
        $this->importStats = [
            'processed' => 0,
            'imported' => 0,
            'updated' => 0,
            'skipped' => 0,
            'errors' => []
        ];
    }

    public function chunkSize(): int
    {
        return 150;
    }

    /**
     * @param array $row
     *
     * @return Contact|null
     */
    public function model(array $row)
    {
        $this->importStats['processed']++;
        
        try {
            // Validate required fields
            if (empty($row['name']) || empty($row['phone'])) {
                $this->importStats['skipped']++;
                $this->importStats['errors'][] = "Row {$this->importStats['processed']}: Missing name or phone";
                return null;
            }

            // Normalize phone number
            $normalizedPhone = $this->phoneService->normalizePhone($row['phone']);
            
            if (!$this->phoneService->isValidPhone($normalizedPhone)) {
                $this->importStats['skipped']++;
                $this->importStats['errors'][] = "Row {$this->importStats['processed']}: Invalid phone number '{$row['phone']}'";
                return null;
            }

            $keys = array_keys($row);
            $keysForFields = [];
            foreach ($keys as $key => $value) {
                $keysForFields[$key] = $this->getOrMakeField($value);
            }

            // Check if contact already exists
            $prevContact = Contact::where('phone', $normalizedPhone)
                ->where('company_id', session('company_id'))
                ->first();
                
            if ($prevContact) {
                $contact = $prevContact;
                // Detach all fields from the contact
                $contact->fields()->detach();
                $this->importStats['updated']++;
            } else {
                $contact = new Contact([
                    'name' => trim($row['name']),
                    'phone' => $normalizedPhone,
                    'company_id' => session('company_id')
                ]);
                $contact->save();
                $this->importStats['imported']++;
            }

            // Update avatar if provided
            if (isset($row['avatar']) && !empty($row['avatar'])) {
                $contact->avatar = $row['avatar'];
            }

            // Update email if provided
            if (isset($row['email']) && !empty($row['email'])) {
                $contact->email = $row['email'];
            }

            // Attach custom fields
            foreach ($keysForFields as $key => $fieldID) {
                if ($fieldID != 0 && !empty($row[$keys[$key]])) {
                    $contact->fields()->attach($fieldID, ['value' => $row[$keys[$key]]]);
                }
            }
            
            $contact->update();

            return $contact;
            
        } catch (\Exception $e) {
            $this->importStats['skipped']++;
            $this->importStats['errors'][] = "Row {$this->importStats['processed']}: " . $e->getMessage();
            Log::error('Contact import error', [
                'row' => $row,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    private function getOrMakeField($field_name){
        if($field_name=="name"||$field_name=="phone"||$field_name=="avatar"||$field_name=="email"){
            return 0;
        }
        $field=Field::where('name',$field_name)
            ->where('company_id', session('company_id'))
            ->first();
        if(!$field){
            $field=Field::create([
                'name'     => $field_name,
                'type'=>"text",
                'company_id' => session('company_id')
            ]);
            $field->save();
        }
        return $field->id;
    }

    /**
     * Get import statistics
     */
    public function getImportStats(): array
    {
        return $this->importStats;
    }

    /**
     * Reset import statistics
     */
    public function resetStats(): void
    {
        $this->importStats = [
            'processed' => 0,
            'imported' => 0,
            'updated' => 0,
            'skipped' => 0,
            'errors' => []
        ];
    }
}