<?php

namespace Modules\Contacts\Models;

use App\Models\Company;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Scopes\CompanyScope;

class Contact extends Model
{
    use SoftDeletes;
    
    protected $table = 'contacts';
    public $guarded = [];

    public function groups()
    {
        return $this->belongsToMany(
                Group::class,
                'groups_contacts',
                'contact_id',
                'group_id'
            );
    }

    public function getCompany(){
        return Company::findOrFail($this->company_id);
    }

    public function fields()
    {
        return $this->belongsToMany(
                Field::class,
                'custom_contacts_fields_contacts',
                'contact_id',
                'custom_contacts_field_id'
            )->withPivot('value');
    }

   

    public function country()
    {
        return $this->belongsTo(
                Country::class
            );
    }

    /**
     * Get the agent (user) assigned to this contact.
     */
    public function assignedAgent()
    {
        return $this->belongsTo(\App\Models\User::class, 'user_id');
    }

    /**
     * Check if contact is assigned to a specific agent.
     */
    public function isAssignedTo($userId): bool
    {
        return $this->user_id == $userId;
    }

    /**
     * Check if contact is assigned to any agent.
     */
    public function isAssigned(): bool
    {
        return !is_null($this->user_id);
    }

    /**
     * Assign contact to an agent.
     */
    public function assignToAgent($userId)
    {
        $this->user_id = $userId;
        $this->save();
        return $this;
    }

    /**
     * Unassign contact from agent.
     */
    public function unassignAgent()
    {
        $this->user_id = null;
        $this->save();
        return $this;
    }

    /**
     * Get the CRM mappings for this contact.
     */
    public function crmMappings()
    {
        return $this->hasMany(ContactCrmMapping::class);
    }

    /**
     * Get the CRM connections through mappings.
     */
    public function crmConnections()
    {
        return $this->belongsToMany(
            CrmConnection::class,
            'contact_crm_mapping',
            'contact_id',
            'crm_connection_id'
        )->withPivot(['crm_contact_id', 'sync_status', 'crm_data', 'last_synced_at', 'sync_error']);
    }

    /**
     * Check if contact is synced with a specific CRM connection.
     */
    public function isSyncedWithCrm($crmConnectionId): bool
    {
        return $this->crmMappings()
            ->where('crm_connection_id', $crmConnectionId)
            ->where('sync_status', 'synced')
            ->exists();
    }

    /**
     * Get the CRM mapping for a specific connection.
     */
    public function getCrmMapping($crmConnectionId)
    {
        return $this->crmMappings()
            ->where('crm_connection_id', $crmConnectionId)
            ->first();
    }

    /**
     * Check if this contact is a company.
     */
    public function isCompany(): bool
    {
        return $this->contact_type === 'company';
    }

    /**
     * Check if this contact is an individual.
     */
    public function isIndividual(): bool
    {
        return $this->contact_type === 'individual';
    }

    /**
     * Get the contact type display name.
     */
    public function getContactTypeDisplayAttribute(): string
    {
        return $this->isCompany() ? 'Company' : 'Individual';
    }

    /**
     * Get company-specific fields for display.
     */
    public function getCompanyFieldsAttribute(): array
    {
        if (!$this->isCompany()) {
            return [];
        }

        return [
            'vat_number' => $this->vat_number,
            'function' => $this->function,
        ];
    }

    /**
     * Get individual-specific fields for display.
     */
    public function getIndividualFieldsAttribute(): array
    {
        if (!$this->isIndividual()) {
            return [];
        }

        return [
            'title' => $this->title,
            'function' => $this->function,
            'language' => $this->language,
            'timezone' => $this->timezone,
        ];
    }

    protected static function booted(){
        static::addGlobalScope(new CompanyScope);

        static::creating(function ($model){
           $company_id=session('company_id',null);
            if($company_id){
                $model->company_id=$company_id;
            }
        });

        static::created(function ($model){
            //Determine the country
            $country_id=$model->getCountryByPhoneNumber($model->phone);
            if($country_id){
                $model->country_id=$country_id;
                $model->update();
            }
            
         });
    }
    
    private function getCountryByPhoneNumber($phoneNumber) {

        if (strpos($phoneNumber, '+') !== 0) {
            $phoneNumber = '+' . $phoneNumber;
        }

        $prefixes = Country::pluck('id','phone_code');
       
        // Use regular expression to extract the prefix
        if (preg_match('/^\+(\d{4})/', $phoneNumber, $matches)) {
            $prefix = $matches[1];
    
            // Check if the prefix exists in the array
            if (isset($prefixes[$prefix])) {
                return $prefixes[$prefix];
                }else if (preg_match('/^\+(\d{3})/', $phoneNumber, $matches)) {
                    $prefix = $matches[1];
            
                    // Check if the prefix exists in the array
                    if (isset($prefixes[$prefix])) {
                        return $prefixes[$prefix];
                    }else if (preg_match('/^\+(\d{2})/', $phoneNumber, $matches)) {
                        $prefix = $matches[1];
                
                        // Check if the prefix exists in the array
                        if (isset($prefixes[$prefix])) {
                            return $prefixes[$prefix];
                        }else if (preg_match('/^\+(\d{1})/', $phoneNumber, $matches)) {
                        $prefix = $matches[1];
                
                        // Check if the prefix exists in the array
                        if (isset($prefixes[$prefix])) {
                            return $prefixes[$prefix];
                        }
                    }
                    }   
                }
            }
    
        return null;
    }
}
