<?php

namespace Modules\Contacts\Services\CrmIntegration\Connectors;

use Modules\Contacts\Services\CrmIntegration\CrmConnectorInterface;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class HubSpotConnector implements CrmConnectorInterface
{
    /**
     * HubSpot API base URL.
     */
    private const API_BASE_URL = 'https://api.hubapi.com';

    /**
     * Test the connection to HubSpot.
     *
     * @param array $credentials
     * @return bool
     * @throws \Exception
     */
    public function testConnection(array $credentials): bool
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $credentials['access_token'],
                'Content-Type' => 'application/json',
            ])->get(self::API_BASE_URL . '/crm/v3/objects/contacts', [
                'limit' => 1,
            ]);

            if ($response->successful()) {
                return true;
            }

            Log::error('HubSpot connection test failed', [
                'status' => $response->status(),
                'response' => $response->body(),
            ]);

            return false;
        } catch (\Exception $e) {
            Log::error('HubSpot connection test error', [
                'error' => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    /**
     * Get contacts from HubSpot.
     *
     * @param array $credentials
     * @param array $options
     * @return array
     * @throws \Exception
     */
    public function getContacts(array $credentials, array $options = []): array
    {
        try {
            $limit = $options['limit'] ?? 100;
            $after = $options['after'] ?? null;
            $properties = $options['properties'] ?? ['firstname', 'lastname', 'email', 'phone', 'company'];

            $params = [
                'limit' => $limit,
                'properties' => implode(',', $properties),
            ];

            if ($after) {
                $params['after'] = $after;
            }

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $credentials['access_token'],
                'Content-Type' => 'application/json',
            ])->get(self::API_BASE_URL . '/crm/v3/objects/contacts', $params);

            if (!$response->successful()) {
                throw new \Exception('Failed to fetch contacts from HubSpot: ' . $response->body());
            }

            $data = $response->json();

            return [
                'contacts' => $data['results'] ?? [],
                'pagination' => [
                    'has_more' => isset($data['paging']['next']),
                    'after' => $data['paging']['next']['after'] ?? null,
                ],
            ];
        } catch (\Exception $e) {
            Log::error('HubSpot get contacts error', [
                'error' => $e->getMessage(),
                'options' => $options,
            ]);

            throw $e;
        }
    }

    /**
     * Get a single contact from HubSpot.
     *
     * @param array $credentials
     * @param string $contactId
     * @return array|null
     * @throws \Exception
     */
    public function getContact(array $credentials, string $contactId): ?array
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $credentials['access_token'],
                'Content-Type' => 'application/json',
            ])->get(self::API_BASE_URL . '/crm/v3/objects/contacts/' . $contactId);

            if ($response->status() === 404) {
                return null;
            }

            if (!$response->successful()) {
                throw new \Exception('Failed to fetch contact from HubSpot: ' . $response->body());
            }

            return $response->json();
        } catch (\Exception $e) {
            Log::error('HubSpot get contact error', [
                'error' => $e->getMessage(),
                'contact_id' => $contactId,
            ]);

            throw $e;
        }
    }

    /**
     * Create a contact in HubSpot.
     *
     * @param array $credentials
     * @param array $contactData
     * @return array
     * @throws \Exception
     */
    public function createContact(array $credentials, array $contactData): array
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $credentials['access_token'],
                'Content-Type' => 'application/json',
            ])->post(self::API_BASE_URL . '/crm/v3/objects/contacts', [
                'properties' => $contactData,
            ]);

            if (!$response->successful()) {
                throw new \Exception('Failed to create contact in HubSpot: ' . $response->body());
            }

            return $response->json();
        } catch (\Exception $e) {
            Log::error('HubSpot create contact error', [
                'error' => $e->getMessage(),
                'contact_data' => $contactData,
            ]);

            throw $e;
        }
    }

    /**
     * Update a contact in HubSpot.
     *
     * @param array $credentials
     * @param string $contactId
     * @param array $contactData
     * @return array
     * @throws \Exception
     */
    public function updateContact(array $credentials, string $contactId, array $contactData): array
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $credentials['access_token'],
                'Content-Type' => 'application/json',
            ])->patch(self::API_BASE_URL . '/crm/v3/objects/contacts/' . $contactId, [
                'properties' => $contactData,
            ]);

            if (!$response->successful()) {
                throw new \Exception('Failed to update contact in HubSpot: ' . $response->body());
            }

            return $response->json();
        } catch (\Exception $e) {
            Log::error('HubSpot update contact error', [
                'error' => $e->getMessage(),
                'contact_id' => $contactId,
                'contact_data' => $contactData,
            ]);

            throw $e;
        }
    }

    /**
     * Delete a contact from HubSpot.
     *
     * @param array $credentials
     * @param string $contactId
     * @return bool
     * @throws \Exception
     */
    public function deleteContact(array $credentials, string $contactId): bool
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $credentials['access_token'],
            ])->delete(self::API_BASE_URL . '/crm/v3/objects/contacts/' . $contactId);

            if (!$response->successful()) {
                throw new \Exception('Failed to delete contact from HubSpot: ' . $response->body());
            }

            return true;
        } catch (\Exception $e) {
            Log::error('HubSpot delete contact error', [
                'error' => $e->getMessage(),
                'contact_id' => $contactId,
            ]);

            throw $e;
        }
    }

    /**
     * Get available fields from HubSpot.
     *
     * @param array $credentials
     * @return array
     * @throws \Exception
     */
    public function getAvailableFields(array $credentials): array
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $credentials['access_token'],
                'Content-Type' => 'application/json',
            ])->get(self::API_BASE_URL . '/crm/v3/properties/contacts');

            if (!$response->successful()) {
                throw new \Exception('Failed to fetch fields from HubSpot: ' . $response->body());
            }

            $data = $response->json();
            $fields = [];

            // Define essential fields for contact mapping
            $essentialFields = [
                'firstname', 'lastname', 'email', 'phone', 'mobilephone', 'company', 
                'jobtitle', 'job_function', 'department', 'industry', 'website',
                'address', 'city', 'state', 'country', 'zip', 'timezone',
                'lifecyclestage', 'leadstatus', 'hs_lead_status', 'hs_lead_score',
                'createdate', 'lastmodifieddate', 'notes_last_contacted', 'notes_last_activity_date'
            ];

            foreach ($data['results'] as $field) {
                // Only include essential fields to avoid overwhelming the dropdown
                if (in_array($field['name'], $essentialFields)) {
                    $fields[] = [
                        'name' => $field['name'],
                        'label' => $field['label'],
                        'type' => $field['type'],
                        'required' => $field['required'] ?? false,
                        'description' => $field['description'] ?? '',
                    ];
                }
            }

            return $fields;
        } catch (\Exception $e) {
            Log::error('HubSpot get fields error', [
                'error' => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    /**
     * Get the CRM type.
     *
     * @return string
     */
    public function getType(): string
    {
        return 'hubspot';
    }

    /**
     * Get the CRM display name.
     *
     * @return string
     */
    public function getDisplayName(): string
    {
        return 'HubSpot';
    }

    /**
     * Get the required credentials for HubSpot.
     *
     * @return array
     */
    public function getRequiredCredentials(): array
    {
        return [
            'access_token' => 'Access Token',
        ];
    }

    /**
     * Validate credentials format.
     *
     * @param array $credentials
     * @return bool
     */
    public function validateCredentials(array $credentials): bool
    {
        return isset($credentials['access_token']) && !empty($credentials['access_token']);
    }
}