<?php

namespace Modules\Contacts\Services;

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Modules\Contacts\Models\Field;
use Symfony\Component\HttpFoundation\StreamedResponse;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Log;
use InvalidArgumentException;

/**
 * Demo CSV Service
 * 
 * Generates demo CSV files with sample data for contact import
 * following enterprise standards and best practices.
 * 
 * @package Modules\Contacts\Services
 * @author WA-Tawasul Team
 * @version 1.0.0
 */
class DemoCsvService
{
    /**
     * CSV file encoding
     */
    private const CSV_ENCODING = 'UTF-8';
    
    /**
     * BOM for Excel compatibility
     */
    private const CSV_BOM = "\xEF\xBB\xBF";
    
    /**
     * Default sample data count
     */
    private const DEFAULT_SAMPLE_COUNT = 5;

    /**
     * Generate demo CSV content with sample data
     * 
     * @return string
     * @throws InvalidArgumentException
     */
    public function generateDemoCsv(): string
    {
        try {
            $headers = $this->getCsvHeaders();
            $sampleData = $this->getSampleData();
            
            return $this->arrayToCsv($headers, $sampleData);
        } catch (\Exception $e) {
            Log::error('Failed to generate demo CSV', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw new InvalidArgumentException('Failed to generate demo CSV: ' . $e->getMessage());
        }
    }

    /**
     * Get CSV headers including required, optional, and custom fields
     * 
     * @return array
     */
    private function getCsvHeaders(): array
    {
        $baseHeaders = ['name', 'phone', 'email', 'avatar'];
        
        try {
            // Get existing custom fields for the current company
            $companyId = session('company_id');
            if ($companyId) {
                $customFields = Field::where('company_id', $companyId)
                    ->pluck('name')
                    ->toArray();
                
                return array_merge($baseHeaders, $customFields);
            } else {
                Log::warning('No company_id in session, using base headers only');
                return $baseHeaders;
            }
        } catch (\Exception $e) {
            Log::warning('Failed to get custom fields, using base headers only', [
                'error' => $e->getMessage(),
                'company_id' => session('company_id')
            ]);
            return $baseHeaders;
        }
    }

    /**
     * Generate realistic sample data with various phone formats
     * 
     * @return array
     */
    private function getSampleData(): array
    {
        return [
            [
                'name' => 'John Doe',
                'phone' => '+971 55 869 4287', // UAE format with spaces
                'email' => 'john.doe@example.com',
                'avatar' => 'https://example.com/avatars/john.jpg',
                'position' => 'Manager',
                'department' => 'Sales',
                'status' => 'Active'
            ],
            [
                'name' => 'Jane Smith',
                'phone' => '971558694321', // UAE format without spaces
                'email' => 'jane.smith@example.com',
                'avatar' => '',
                'position' => 'Developer',
                'department' => 'Engineering',
                'status' => 'Active'
            ],
            [
                'name' => 'Bob Johnson',
                'phone' => '+1 (555) 123-4567', // US format with formatting
                'email' => 'bob.johnson@example.com',
                'avatar' => '',
                'position' => 'Designer',
                'department' => 'Creative',
                'status' => 'Pending'
            ],
            [
                'name' => 'Sarah Wilson',
                'phone' => '966501234567', // Saudi format
                'email' => 'sarah.wilson@example.com',
                'avatar' => '',
                'position' => 'Analyst',
                'department' => 'Marketing',
                'status' => 'Active'
            ],
            [
                'name' => 'Ahmed Al-Rashid',
                'phone' => '+965 1234 5678', // Kuwait format
                'email' => 'ahmed.alrashid@example.com',
                'avatar' => '',
                'position' => 'Consultant',
                'department' => 'Business',
                'status' => 'Active'
            ]
        ];
    }

    /**
     * Convert array data to CSV format
     * 
     * @param array $headers
     * @param array $data
     * @return string
     * @throws InvalidArgumentException
     */
    private function arrayToCsv(array $headers, array $data): string
    {
        $output = fopen('php://temp', 'r+');
        
        if ($output === false) {
            throw new InvalidArgumentException('Failed to create temporary file for CSV generation');
        }
        
        try {
            // Add BOM for Excel compatibility
            fwrite($output, self::CSV_BOM);
            
            // Write headers
            fputcsv($output, $headers);
            
            // Write data rows
            foreach ($data as $row) {
                $csvRow = [];
                foreach ($headers as $header) {
                    $csvRow[] = $row[$header] ?? '';
                }
                fputcsv($output, $csvRow);
            }
            
            rewind($output);
            $csv = stream_get_contents($output);
            
            if ($csv === false) {
                throw new InvalidArgumentException('Failed to read CSV content from temporary file');
            }
            
            return $csv;
        } finally {
            fclose($output);
        }
    }

    /**
     * Generate and return CSV as downloadable response
     * 
     * @return Response
     * @throws InvalidArgumentException
     */
    public function downloadDemoCsv(): Response
    {
        try {
            $csv = $this->generateDemoCsv();
            $filename = $this->generateFilename();
            
            return response($csv, 200, $this->getResponseHeaders($filename, strlen($csv)));
        } catch (\Exception $e) {
            Log::error('Failed to create CSV download response', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw new InvalidArgumentException('Failed to create CSV download: ' . $e->getMessage());
        }
    }
    
    /**
     * Generate filename for CSV download
     * 
     * @return string
     */
    private function generateFilename(): string
    {
        $date = now()->format('Y-m-d');
        $companyId = session('company_id', 'default');
        
        return "contacts_import_template_{$companyId}_{$date}.csv";
    }
    
    /**
     * Get response headers for CSV download
     * 
     * @param string $filename
     * @param int $contentLength
     * @return array
     */
    private function getResponseHeaders(string $filename, int $contentLength): array
    {
        return [
            'Content-Type' => 'text/csv; charset=' . self::CSV_ENCODING,
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
            'Content-Length' => $contentLength,
            'Cache-Control' => 'no-cache, must-revalidate',
            'Pragma' => 'no-cache',
            'Expires' => '0'
        ];
    }

    /**
     * Get CSV template with company-specific custom fields
     * 
     * @return array
     */
    public function getCompanySpecificTemplate(): array
    {
        try {
            $headers = $this->getCsvHeaders();
            $sampleData = $this->getSampleData();
            
            return [
                'headers' => $headers,
                'sample_data' => $sampleData,
                'required_fields' => ['name', 'phone'],
                'optional_fields' => ['email', 'avatar'],
                'custom_fields' => array_diff($headers, ['name', 'phone', 'email', 'avatar']),
                'phone_formats' => [
                    'examples' => [
                        '+971 55 869 4287',
                        '971558694287',
                        '+1 (555) 123-4567',
                        '966501234567',
                        '+965 1234 5678'
                    ],
                    'note' => 'Phone numbers will be automatically normalized to international format without spaces'
                ],
                'generated_at' => now()->toISOString(),
                'company_id' => session('company_id')
            ];
        } catch (\Exception $e) {
            Log::error('Failed to get company specific template', [
                'error' => $e->getMessage(),
                'company_id' => session('company_id')
            ]);
            
            // Return basic template as fallback
            return [
                'headers' => ['name', 'phone', 'email', 'avatar'],
                'sample_data' => $this->getSampleData(),
                'required_fields' => ['name', 'phone'],
                'optional_fields' => ['email', 'avatar'],
                'custom_fields' => [],
                'phone_formats' => [
                    'examples' => ['+971 55 869 4287', '971558694287'],
                    'note' => 'Phone numbers will be automatically normalized'
                ],
                'generated_at' => now()->toISOString(),
                'company_id' => session('company_id'),
                'error' => 'Failed to load custom fields'
            ];
        }
    }
    
    /**
     * Validate CSV data structure
     * 
     * @param array $data
     * @return bool
     */
    public function validateCsvData(array $data): bool
    {
        if (empty($data)) {
            return false;
        }
        
        $requiredFields = ['name', 'phone'];
        
        foreach ($data as $row) {
            foreach ($requiredFields as $field) {
                if (!isset($row[$field]) || empty(trim($row[$field]))) {
                    return false;
                }
            }
        }
        
        return true;
    }
    
    /**
     * Get CSV statistics
     * 
     * @param array $data
     * @return array
     */
    public function getCsvStatistics(array $data): array
    {
        $totalRows = count($data);
        $validRows = 0;
        $invalidRows = 0;
        $fields = [];
        
        foreach ($data as $row) {
            $hasRequiredFields = !empty($row['name']) && !empty($row['phone']);
            
            if ($hasRequiredFields) {
                $validRows++;
            } else {
                $invalidRows++;
            }
            
            $fields = array_merge($fields, array_keys($row));
        }
        
        return [
            'total_rows' => $totalRows,
            'valid_rows' => $validRows,
            'invalid_rows' => $invalidRows,
            'unique_fields' => array_unique($fields),
            'field_count' => count(array_unique($fields))
        ];
    }
}