<?php
namespace Modules\Flowmaker\Models\Nodes;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Modules\Flowmaker\Models\Contact;

class NewRequest extends Node
{
    public function process($message, $data)
    {
        Log::info('🔄 Processing NewRequest node', [
            'node_id' => $this->id ?? 'unknown',
            'flow_id' => $this->flow_id ?? 'unknown'
        ]);
        
        try {
            // Get request settings from node data
            $nodeData = $this->getDataAsArray();
            $requestSettings = $nodeData['settings']['new_request'] ?? [];
            
            Log::debug('📋 Node settings', ['settings' => $requestSettings]);

            // Find the contact
            $contactId = is_object($data) ? $data->contact_id : $data['contact_id'];
            $contact = Contact::find($contactId);
            
            if (!$contact) {
                Log::error('Contact not found', ['contactId' => $contactId]);
                return ['success' => false];
            }

            Log::debug('Contact found', ['contact' => $contact->id]);

            // Extract settings
            $method = strtoupper($requestSettings['method'] ?? 'GET');
            $url = $requestSettings['url'] ?? '';
            $headersJson = $requestSettings['headers'] ?? '{}';
            $body = $requestSettings['body'] ?? '';
            $bodyType = $requestSettings['bodyType'] ?? 'json';
            $responseVar = $requestSettings['responseVar'] ?? '';

            // Transform URL with variables
            $originalUrl = $url;
            $url = $contact->changeVariables($url, $this->flow_id);
            Log::debug('🔗 Interpolated URL', [
                'original' => $originalUrl,
                'interpolated' => $url
            ]);

            if (empty($url)) {
                Log::error('URL is empty after transformation');
                return ['success' => false];
            }

            // Parse and process headers
            $requestHeaders = $this->processHeaders($headersJson, $contact);

            // Prepare request data
            $requestData = $this->prepareRequestData($requestSettings, $contact);

            // Handle Basic Auth if credentials are provided
            $basicAuthHeader = $this->handleBasicAuth($requestSettings, $contact);
            if ($basicAuthHeader) {
                $requestHeaders['Authorization'] = $basicAuthHeader;
                Log::info('🔐 Basic Auth header added', ['auth_header' => $basicAuthHeader]);
            }

            Log::info('🌐 Making HTTP request', [
                'method' => $method,
                'url' => $url,
                'headers' => $requestHeaders,
                'body_type' => $bodyType,
                'timeout' => $requestSettings['timeout'] ?? 10,
                'retries' => $requestSettings['retry'] ?? 0
            ]);

            // Make HTTP request based on method
            $response = null;
            $httpClient = Http::withHeaders($requestHeaders)->timeout($requestSettings['timeout'] ?? 10);
            
            switch ($method) {
                case 'GET':
                    $response = $httpClient->get($url, $requestData);
                    break;
                case 'POST':
                    if ($this->isJsonRequest($bodyType)) {
                        $response = $httpClient->withHeaders(['Content-Type' => 'application/json'])->post($url, $requestData);
                    } else {
                        $response = $httpClient->post($url, $requestData);
                    }
                    break;
                case 'PUT':
                    if ($this->isJsonRequest($bodyType)) {
                        $response = $httpClient->withHeaders(['Content-Type' => 'application/json'])->put($url, $requestData);
                    } else {
                        $response = $httpClient->put($url, $requestData);
                    }
                    break;
                case 'DELETE':
                    $response = $httpClient->delete($url, $requestData);
                    break;
                default:
                    Log::error('Unsupported HTTP method', ['method' => $method]);
                    return ['success' => false];
            }

            // Log response
            if ($response) {
                $statusCode = $response->status();
                $isSuccess = $statusCode >= 200 && $statusCode < 300;
                
                Log::info($isSuccess ? '✅ Request successful' : '⚠️ Request completed with non-success status', [
                    'status' => $statusCode,
                    'response_var' => $responseVar,
                    'is_success' => $isSuccess
                ]);
            }

            if ($response) {
                $responseData = [
                    'status' => $response->status(),
                    'body' => $response->body(),
                    'json' => $response->json(),
                    'headers' => $response->headers()
                ];

                // Store response in variable if specified
                if (!empty($responseVar)) {
                    $contact->setContactState($this->flow_id, $responseVar, json_encode($responseData));
                    $contact->setContactState($this->flow_id, $responseVar . '_status', $response->status());
                    $contact->setContactState($this->flow_id, $responseVar . '_body', $response->body());
                    
                    // Debug log for variable storage
                    Log::info('💾 Stored response variables', [
                        'flow_id' => $this->flow_id,
                        'response_var' => $responseVar,
                        'variables_stored' => [
                            $responseVar,
                            $responseVar . '_status',
                            $responseVar . '_body',
                            $responseVar . '_json',
                            $responseVar . '_products'
                        ]
                    ]);
                    
                    if ($response->json()) {
                        $contact->setContactState($this->flow_id, $responseVar . '_json', json_encode($response->json()));
                        
                        // Store simplified product data for easier access
                        $jsonData = $response->json();
                        if ($this->isWooCommerceProductsResponse($jsonData)) {
                            $simplifiedProducts = $this->extractSimplifiedProducts($jsonData);
                            $contact->setContactState($this->flow_id, $responseVar . '_products', json_encode($simplifiedProducts));
                            
                            // Debug log to see what's being stored
                            Log::info('📦 Stored simplified products', [
                                'flow_id' => $this->flow_id,
                                'response_var' => $responseVar,
                                'products_count' => count($simplifiedProducts),
                                'first_product' => $simplifiedProducts[0] ?? 'none'
                            ]);
                        }
                    }
                }

                // Format response for display
                $formattedResponse = $this->formatResponse($response, $requestSettings);
                Log::debug('📄 Formatted response', ['formatted' => $formattedResponse]);
                
                // Store the formatted response as the main response variable
                if (!empty($responseVar)) {
                    $contact->setContactState($this->flow_id, $responseVar, $formattedResponse);
                }

            } else {
                Log::error('❌ NewRequest failed - no response');
                return ['success' => false];
            }

        } catch (\Exception $e) {
            Log::error('Error processing NewRequest', ['error' => $e->getMessage()]);
            return ['success' => false];
        }

        // Continue flow to next node if one exists (moved outside try-catch)
        // Only continue if the API call was successful (status 2xx)
        $shouldContinue = true;
        if ($response) {
            $statusCode = $response->status();
            $shouldContinue = $statusCode >= 200 && $statusCode < 300;
            
            if (!$shouldContinue) {
                Log::warning('⚠️ Not continuing flow due to non-success status', [
                    'current_node' => $this->id,
                    'status_code' => $statusCode
                ]);
            }
        }
        
        if ($shouldContinue) {
            try {
                $nextNode = $this->getNextNodeId();
                if ($nextNode) {
                    Log::info('🔄 Continuing to next node', [
                        'current_node' => $this->id,
                        'next_node' => $nextNode->id,
                        'next_node_type' => get_class($nextNode)
                    ]);
                    
                    // Process the next node
                    $result = $nextNode->process($message, $data);
                    Log::info('✅ Next node processed successfully', [
                        'current_node' => $this->id,
                        'next_node' => $nextNode->id,
                        'result' => $result
                    ]);
                } else {
                    Log::warning('⚠️ No next node found', ['current_node' => $this->id]);
                }
            } catch (\Exception $e) {
                Log::error('Error continuing to next node', [
                    'current_node' => $this->id,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }

        return ['success' => true];
    }

    /**
     * Process headers JSON string and handle sensitive headers
     */
    protected function processHeaders($headersJson, $contact)
    {
        try {
            $headers = json_decode($headersJson, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                Log::error('❌ Invalid headers JSON', ['error' => json_last_error_msg()]);
                return [];
            }

            $requestHeaders = [];
            foreach ($headers as $key => $value) {
                // Special handling for sensitive headers
                $processedValue = $this->processHeaderValue($value, $contact);
                
                // Log the header processing for debugging
                if ($key === 'Authorization') {
                    Log::debug('🔐 Processing Authorization header', [
                        'original' => $value,
                        'processed' => $processedValue,
                        'length' => strlen($processedValue)
                    ]);
                }
                
                $requestHeaders[$key] = $processedValue;
            }

            return $requestHeaders;
        } catch (\Exception $e) {
            Log::error('❌ Error processing headers', ['error' => $e->getMessage()]);
            return [];
        }
    }

    /**
     * Handle Basic Auth credentials from request settings
     */
    protected function handleBasicAuth($requestSettings, $contact)
    {
        $username = $requestSettings['basicAuthUsername'] ?? '';
        $password = $requestSettings['basicAuthPassword'] ?? '';
        
        if (!empty($username) && !empty($password)) {
            // Interpolate variables in credentials if needed
            $username = $contact->changeVariables($username, $this->flow_id);
            $password = $contact->changeVariables($password, $this->flow_id);
            
            // Create Basic Auth header
            $credentials = $username . ':' . $password;
            $base64Credentials = base64_encode($credentials);
            
            Log::debug('🔐 Creating Basic Auth header', [
                'username' => $username,
                'password_length' => strlen($password),
                'base64_length' => strlen($base64Credentials)
            ]);
            
            return 'Basic ' . $base64Credentials;
        }
        
        return null;
    }

    /**
     * Prepare request data based on body type and settings
     */
    protected function prepareRequestData($requestSettings, $contact)
    {
        $bodyType = $requestSettings['bodyType'] ?? 'json';
        $body = $requestSettings['body'] ?? '';
        $queryParams = $requestSettings['queryParams'] ?? [];

        $requestData = [];

        // Process query parameters
        if (!empty($queryParams) && is_array($queryParams)) {
            foreach ($queryParams as $param) {
                if (!empty($param['key']) && isset($param['value'])) {
                    $key = $contact->changeVariables($param['key'], $this->flow_id);
                    $value = $contact->changeVariables($param['value'], $this->flow_id);
                    $requestData[$key] = $value;
                }
            }
        }

        // Process body for POST/PUT requests
        if (in_array(strtoupper($requestSettings['method'] ?? 'GET'), ['POST', 'PUT', 'PATCH'])) {
            if ($this->isJsonRequest($bodyType)) {
                try {
                    $bodyData = json_decode($body, true);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        // Interpolate variables in JSON body
                        $bodyData = $this->interpolateJsonVariables($bodyData, $contact);
                        $requestData = array_merge($requestData, $bodyData);
                    }
                } catch (\Exception $e) {
                    Log::error('❌ Error processing JSON body', ['error' => $e->getMessage()]);
                }
            } else {
                // For form data, process as key-value pairs
                $lines = explode("\n", $body);
                foreach ($lines as $line) {
                    $parts = explode('=', $line, 2);
                    if (count($parts) === 2) {
                        $key = trim($parts[0]);
                        $value = trim($parts[1]);
                        $key = $contact->changeVariables($key, $this->flow_id);
                        $value = $contact->changeVariables($value, $this->flow_id);
                        $requestData[$key] = $value;
                    }
                }
            }
        }

        return $requestData;
    }

    /**
     * Check if request should be sent as JSON
     */
    protected function isJsonRequest($bodyType)
    {
        return $bodyType === 'json';
    }

    /**
     * Interpolate variables in JSON data recursively
     */
    protected function interpolateJsonVariables($data, $contact)
    {
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                if (is_string($value)) {
                    $data[$key] = $contact->changeVariables($value, $this->flow_id);
                } elseif (is_array($value)) {
                    $data[$key] = $this->interpolateJsonVariables($value, $contact);
                }
            }
        }
        return $data;
    }

    /**
     * Process header value with special handling for sensitive headers
     */
    protected function processHeaderValue($value, $contact)
    {
        // Check if this looks like a sensitive header value
        if ($this->isSensitiveHeaderValue($value)) {
            // For sensitive values, only interpolate if explicitly using variables
            if (strpos($value, '{{') !== false) {
                $processedValue = $contact->changeVariables($value, $this->flow_id);
                // Ensure Base64 padding is correct after interpolation
                return $this->ensureBase64Padding($processedValue);
            }
            // Otherwise, use the raw value with proper padding
            return $this->ensureBase64Padding($value);
        }
        
        // For non-sensitive values, always interpolate variables
        return $contact->changeVariables($value, $this->flow_id);
    }

    /**
     * Ensure Base64 string has proper padding
     */
    protected function ensureBase64Padding($value)
    {
        // Check if this is a Basic Authorization header
        if (preg_match('/^Basic (.+)$/', $value, $matches)) {
            $base64Part = $matches[1];
            
            // Add padding if needed
            $padding = strlen($base64Part) % 4;
            if ($padding) {
                $base64Part .= str_repeat('=', 4 - $padding);
            }
            
            // Validate Base64 format
            if (preg_match('/^[A-Za-z0-9+\/]+={0,2}$/', $base64Part)) {
                return 'Basic ' . $base64Part;
            }
        }
        
        return $value;
    }

    /**
     * Create proper WooCommerce Basic Auth token
     */
    protected function createWooCommerceAuthToken($consumerKey, $consumerSecret)
    {
        // Clean the keys (remove any whitespace)
        $consumerKey = trim($consumerKey);
        $consumerSecret = trim($consumerSecret);
        
        // Create the credentials string
        $credentials = $consumerKey . ':' . $consumerSecret;
        
        // Encode to Base64
        $base64Credentials = base64_encode($credentials);
        
        // Ensure proper padding
        $padding = strlen($base64Credentials) % 4;
        if ($padding) {
            $base64Credentials .= str_repeat('=', 4 - $padding);
        }
        
        return 'Basic ' . $base64Credentials;
    }

    /**
     * Validate WooCommerce credentials
     */
    protected function validateWooCommerceCredentials($consumerKey, $consumerSecret)
    {
        // Check if keys follow WooCommerce pattern
        if (!preg_match('/^ck_[a-zA-Z0-9]+$/', $consumerKey)) {
            Log::error('❌ Invalid WooCommerce Consumer Key format', ['key' => $consumerKey]);
            return false;
        }
        
        if (!preg_match('/^cs_[a-zA-Z0-9]+$/', $consumerSecret)) {
            Log::error('❌ Invalid WooCommerce Consumer Secret format', ['secret' => $consumerSecret]);
            return false;
        }
        
        return true;
    }

    /**
     * Check if a header value is sensitive and should be protected
     */
    protected function isSensitiveHeaderValue($value)
    {
        // Check for Basic Authorization patterns (more flexible)
        if (preg_match('/^Basic [A-Za-z0-9+\/]+={0,2}$/', $value)) {
            return true;
        }
        
        // Check for Bearer token patterns
        if (preg_match('/^Bearer [A-Za-z0-9\-._~+\/=]+$/', $value)) {
            return true;
        }
        
        // Check for API key patterns (WooCommerce, Stripe, etc.)
        if (preg_match('/^(ck_|cs_|pk_|sk_|rk_)[a-zA-Z0-9_]+$/', $value)) {
            return true;
        }
        
        // Check for long random strings (likely tokens)
        if (strlen($value) > 50 && preg_match('/^[A-Za-z0-9+\/=_-]+$/', $value)) {
            return true;
        }
        
        return false;
    }

    /**
     * Format response for display
     */
    protected function formatResponse($response, $requestSettings)
    {
        $formatType = $requestSettings['formatType'] ?? 'json';
        $formatTemplate = $requestSettings['formatTemplate'] ?? '';
        
        if ($formatType === 'custom' && !empty($formatTemplate)) {
            // Use custom template
            $jsonData = $response->json();
            if ($jsonData) {
                // Replace variables in template
                $formatted = $formatTemplate;
                foreach ($jsonData as $key => $value) {
                    $formatted = str_replace('{{' . $key . '}}', $value, $formatted);
                }
                return $this->limitResponseForWhatsApp($formatted);
            }
        }
        
        // Default JSON formatting
        $jsonData = $response->json();
        if ($jsonData && is_array($jsonData)) {
            // Check if this is WooCommerce products response
            if ($this->isWooCommerceProductsResponse($jsonData)) {
                return $this->formatWooCommerceProducts($jsonData);
            }
            
            // Generic array formatting
            $formatted = "🛒 Response Data:\n";
            $maxItems = 15; // Limit items for WhatsApp
            $itemsShown = 0;
            
            foreach ($jsonData as $index => $item) {
                if ($itemsShown >= $maxItems) {
                    $remainingCount = count($jsonData) - $maxItems;
                    $formatted .= "\n📋 *... and {$remainingCount} more items found.*\n";
                    break;
                }
                
                if (is_array($item)) {
                    $formatted .= ($index + 1) . ". " . ($item['name'] ?? $item['title'] ?? 'Unknown') . "\n";
                } else {
                    $formatted .= ($index + 1) . ". " . $item . "\n";
                }
                $itemsShown++;
            }
            
            return $this->limitResponseForWhatsApp($formatted);
        }
        
        $formatted = "Response: " . $response->body();
        return $this->limitResponseForWhatsApp($formatted);
    }

    /**
     * Check if response is WooCommerce products data
     */
    protected function isWooCommerceProductsResponse($jsonData)
    {
        if (!is_array($jsonData)) {
            return false;
        }
        
        // Check if it's an array of products (WooCommerce products endpoint)
        if (isset($jsonData[0]) && is_array($jsonData[0])) {
            $firstItem = $jsonData[0];
            return isset($firstItem['id']) && 
                   isset($firstItem['name']) && 
                   isset($firstItem['price']) &&
                   isset($firstItem['type']);
        }
        
        return false;
    }

    /**
     * Format WooCommerce products for display
     */
    protected function formatWooCommerceProducts($products)
    {
        if (empty($products)) {
            return "❌ No products found.";
        }

        $formatted = "🛒 **Products Found:**\n\n";
        $maxProducts = $this->calculateMaxProductsForWhatsApp();
        $productsShown = 0;
        $maxLength = 3800; // Conservative limit (4096 - 296 buffer)
        
        foreach ($products as $index => $product) {
            // Check if we're approaching WhatsApp limit
            if ($productsShown >= $maxProducts) {
                $remainingCount = count($products) - $maxProducts;
                $formatted .= "\n📋 *... and {$remainingCount} more products found.*\n";
                $formatted .= "💡 *Tip: Try a more specific search to see fewer results.*\n";
                break;
            }
            
            $formatted .= "**" . ($index + 1) . ". " . $product['name'] . "**\n";
            $formatted .= "🆔 ID: " . $product['id'] . "\n";
            $formatted .= "💰 Price: $" . $product['price'] . "\n";
            
            // Add description if available (shorter for WhatsApp)
            if (!empty($product['description'])) {
                $description = strip_tags($product['description']);
                $description = substr($description, 0, 40); // Further reduced to 40 characters
                if (strlen($product['description']) > 40) {
                    $description .= "...";
                }
                $formatted .= "📝 " . $description . "\n";
            }
            
            // Add short description if available (shorter for WhatsApp)
            if (!empty($product['short_description'])) {
                $shortDesc = strip_tags($product['short_description']);
                $shortDesc = substr($shortDesc, 0, 35); // Further reduced to 35 characters
                if (strlen($product['short_description']) > 35) {
                    $shortDesc .= "...";
                }
                $formatted .= "📄 " . $shortDesc . "\n";
            }
            
            // Add stock status
            if (isset($product['stock_status'])) {
                $stockStatus = $product['stock_status'] === 'instock' ? '✅ In Stock' : '❌ Out of Stock';
                $formatted .= $stockStatus . "\n";
            }
            
            $formatted .= "\n";
            $productsShown++;
            
            // Check if we're approaching WhatsApp limit (more conservative)
            if (strlen($formatted) > $maxLength) {
                $remainingCount = count($products) - $productsShown;
                $formatted .= "\n📋 *... and {$remainingCount} more products found.*\n";
                $formatted .= "💡 *Tip: Try a more specific search to see fewer results.*\n";
                break;
            }
        }
        
        // Add closing message
        $formatted .= "Would you like to add any of these to your cart?";
        
        // Final safety check - if still over limit, truncate
        if (strlen($formatted) > 4096) {
            $formatted = $this->limitResponseForWhatsApp($formatted, 4096);
        }
        
        Log::debug('📋 Formatted Response Is', [
            'FResponse' => $formatted,
            'length' => strlen($formatted),
            'products_shown' => $productsShown,
            'total_products' => count($products),
            'max_length' => $maxLength,
            'whatsapp_limit' => 4096
        ]);

        return $formatted;
    }

    /**
     * Calculate maximum number of products that can fit in WhatsApp message
     */
    protected function calculateMaxProductsForWhatsApp()
    {
        // More conservative estimate for characters per product: ~120-150 chars
        // Leave more space for header, footer, and buffer
        $estimatedCharsPerProduct = 140;
        $headerFooterChars = 800; // Increased buffer for safety
        $availableChars = 3800 - $headerFooterChars; // Use 3800 as base limit
        
        $maxProducts = floor($availableChars / $estimatedCharsPerProduct);
        
        // Ensure at least 3 products are shown, but max 8 for safety
        return max(3, min($maxProducts, 8));
    }

    /**
     * Get WhatsApp configuration from flow settings
     */
    protected function getWhatsAppConfig()
    {
        $nodeData = $this->getDataAsArray();
        $settings = $nodeData['settings']['new_request'] ?? [];
        
        return [
            'max_length' => $settings['whatsapp_max_length'] ?? 4096,
            'enable_limiting' => $settings['whatsapp_enable_limiting'] ?? true,
            'max_products' => $settings['whatsapp_max_products'] ?? 8,
            'max_items' => $settings['whatsapp_max_items'] ?? 15,
            'truncate_descriptions' => $settings['whatsapp_truncate_descriptions'] ?? true
        ];
    }

    /**
     * Limit any response to WhatsApp character limit
     */
    protected function limitResponseForWhatsApp($response, $maxLength = 4096)
    {
        if (strlen($response) <= $maxLength) {
            return $response;
        }
        
        // Truncate and add indicator
        $truncated = substr($response, 0, $maxLength - 100); // Leave space for truncation message
        
        // Find the last complete line
        $lastNewline = strrpos($truncated, "\n");
        if ($lastNewline !== false) {
            $truncated = substr($truncated, 0, $lastNewline);
        }
        
        $truncated .= "\n\n📋 *Response truncated due to length limit.*\n";
        $truncated .= "💡 *Please refine your search for more specific results.*";
        
        Log::info('📏 Response truncated for WhatsApp', [
            'original_length' => strlen($response),
            'truncated_length' => strlen($truncated),
            'max_length' => $maxLength
        ]);
        
        return $truncated;
    }

    /**
     * Split long response into multiple WhatsApp messages
     */
    protected function splitResponseForWhatsApp($response, $maxLength = 4096)
    {
        if (strlen($response) <= $maxLength) {
            return [$response];
        }
        
        $messages = [];
        $lines = explode("\n", $response);
        $currentMessage = "";
        
        foreach ($lines as $line) {
            // Check if adding this line would exceed limit
            if (strlen($currentMessage . $line . "\n") > $maxLength) {
                if (!empty($currentMessage)) {
                    $messages[] = trim($currentMessage);
                    $currentMessage = "";
                }
                
                // If a single line is too long, truncate it
                if (strlen($line) > $maxLength - 100) {
                    $line = substr($line, 0, $maxLength - 100) . "...";
                }
            }
            
            $currentMessage .= $line . "\n";
        }
        
        // Add the last message if not empty
        if (!empty(trim($currentMessage))) {
            $messages[] = trim($currentMessage);
        }
        
        Log::info('📏 Response split into multiple messages', [
            'original_length' => strlen($response),
            'message_count' => count($messages),
            'max_length' => $maxLength
        ]);
        
        return $messages;
    }

    /**
     * Extract simplified product data for easier template usage
     */
    protected function extractSimplifiedProducts($products)
    {
        $simplified = [];
        
        foreach ($products as $product) {
            $simplified[] = [
                'id' => $product['id'],
                'name' => $product['name'],
                'price' => $product['price'],
                'description' => strip_tags($product['description'] ?? ''),
                'short_description' => strip_tags($product['short_description'] ?? ''),
                'stock_status' => $product['stock_status'] ?? 'unknown',
                'permalink' => $product['permalink'] ?? '',
                'sku' => $product['sku'] ?? '',
                'type' => $product['type'] ?? 'simple'
            ];
        }
        
        return $simplified;
    }

    protected function getNextNodeId($data = null)
    {
        // Get the first outgoing edge's target
        Log::debug('🔍 Getting next node', [
            'current_node' => $this->id,
            'outgoing_edges_count' => count($this->outgoingEdges ?? []),
            'outgoing_edges' => $this->outgoingEdges ?? []
        ]);
        
        if (!empty($this->outgoingEdges)) {
            $nextNode = $this->outgoingEdges[0]->getTarget();
            Log::debug('✅ Next node found', [
                'current_node' => $this->id,
                'next_node' => $nextNode ? $nextNode->id : 'null'
            ]);
            return $nextNode;
        }
        
        // Fallback to parent method if no outgoing edges
        Log::debug('🔄 Falling back to parent getNextNodeId method');
        return parent::getNextNodeId($data);
    }
} 