<?php

namespace Modules\Contacts\Models;

use App\Models\Company;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Scopes\CompanyScope;

class ContactCrmMapping extends Model
{
    use SoftDeletes;

    protected $table = 'contact_crm_mapping';
    public $guarded = [];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'crm_data' => 'array',
        'last_synced_at' => 'datetime',
    ];

    /**
     * Boot the model.
     */
    protected static function booted()
    {
        static::addGlobalScope(new CompanyScope);

        static::creating(function ($model) {
            $company_id = session('company_id', null);
            
            // If no company_id in session, try to get it from the authenticated user
            if (!$company_id && auth()->check() && auth()->user()->company_id) {
                $company_id = auth()->user()->company_id;
            }
            
            // If still no company_id, try to get the first company (for testing purposes)
            if (!$company_id) {
                $company = \App\Models\Company::first();
                if ($company) {
                    $company_id = $company->id;
                }
            }
            
            if ($company_id) {
                $model->company_id = $company_id;
            }
        });
    }

    /**
     * Get the company that owns the mapping.
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * Get the contact that owns the mapping.
     */
    public function contact()
    {
        return $this->belongsTo(Contact::class);
    }

    /**
     * Get the CRM connection for this mapping.
     */
    public function crmConnection()
    {
        return $this->belongsTo(CrmConnection::class);
    }

    /**
     * Check if the mapping is synced.
     */
    public function isSynced(): bool
    {
        return $this->sync_status === 'synced';
    }

    /**
     * Check if the mapping has failed.
     */
    public function hasFailed(): bool
    {
        return $this->sync_status === 'failed';
    }

    /**
     * Check if the mapping is pending.
     */
    public function isPending(): bool
    {
        return $this->sync_status === 'pending';
    }

    /**
     * Check if the mapping has conflicts.
     */
    public function hasConflict(): bool
    {
        return $this->sync_status === 'conflict';
    }

    /**
     * Get the sync status display name.
     */
    public function getSyncStatusDisplayAttribute(): string
    {
        return ucfirst(str_replace('_', ' ', $this->sync_status));
    }

    /**
     * Scope to get only synced mappings.
     */
    public function scopeSynced($query)
    {
        return $query->where('sync_status', 'synced');
    }

    /**
     * Scope to get only failed mappings.
     */
    public function scopeFailed($query)
    {
        return $query->where('sync_status', 'failed');
    }

    /**
     * Scope to get only pending mappings.
     */
    public function scopePending($query)
    {
        return $query->where('sync_status', 'pending');
    }

    /**
     * Scope to get only conflicted mappings.
     */
    public function scopeConflicted($query)
    {
        return $query->where('sync_status', 'conflict');
    }
}