@extends('layouts.app', ['title' => __("Smart Contact Import - WhatsApp Marketing Automation")])

@section('content')
<div class="header pb-8 pt-5 pt-md-8 position-relative overflow-hidden" style="background: linear-gradient(135deg, #25D366 0%, #128C7E 100%, #075E54 100%);">
    <!-- Animated Background Elements -->
    <div class="position-absolute w-100 h-100" style="top: 0; left: 0; z-index: 1;">
        <div class="floating-element floating-element-1"></div>
        <div class="floating-element floating-element-2"></div>
        <div class="floating-element floating-element-3"></div>
        <div class="floating-element floating-element-4"></div>
    </div>
    
    <div class="container-fluid position-relative" style="z-index: 2;">
        <div class="row align-items-center min-vh-30">
            <div class="col-lg-8 mx-auto">
                <div class="text-center text-white header-content">
                    <!-- Badge -->
                    <div class="mb-4">
                        <span class="badge badge-light badge-pill px-4 py-2 mb-3 header-badge">
                            <i class="fas fa-rocket text-success mr-2"></i>
                            <span class="text-dark font-weight-bold">WhatsApp Business Solution</span>
                        </span>
                    </div>
                    
                    <!-- Main Title -->
                    <h1 class="display-3 font-weight-bold mb-4 header-title">
                        <i class="fab fa-whatsapp mr-3 whatsapp-icon"></i>
                        Smart Contact Import
                    </h1>
                    
                    <!-- Subtitle -->
                    <p class="lead mb-4 header-subtitle" style="font-size: 1.3rem; line-height: 1.6; max-width: 600px; margin: 0 auto;">
                        Automate your WhatsApp marketing with intelligent contact management and seamless CRM integration
                    </p>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Bottom Wave -->
    <div class="header-wave">
        <svg viewBox="0 0 1200 120" preserveAspectRatio="none">
            <path d="M0,0V46.29c47.79,22.2,103.59,32.17,158,28,70.36-5.37,136.33-33.31,206.8-37.5C438.64,32.43,512.34,53.67,583,72.05c69.27,18,138.3,24.88,209.4,13.08,36.15-6,69.85-17.84,104.45-29.34C989.49,25,1113-14.29,1200,52.47V0Z" opacity=".25" fill="#ffffff"></path>
            <path d="M0,0V15.81C13,36.92,27.64,56.86,47.69,72.05,99.41,111.27,165,111,224.58,91.58c31.15-10.15,60.09-26.07,89.67-39.8,40.92-19,84.73-46,130.83-49.67,36.26-2.85,70.9,9.42,98.6,31.56,31.77,25.39,62.32,62,103.63,73,40.44,10.79,81.35-6.69,119.13-24.28s75.16-39,116.92-43.05c59.73-5.85,113.28,22.88,168.9,38.84,30.2,8.66,59,6.17,87.09-7.5,22.43-10.89,48-26.93,60.65-49.24V0Z" opacity=".5" fill="#ffffff"></path>
            <path d="M0,0V5.63C149.93,59,314.09,71.32,475.83,42.57c43-7.64,84.23-20.12,127.61-26.46,59-8.63,112.48,12.24,165.56,35.4C827.93,77.22,886,95.24,951.2,90c86.53-7,172.46-45.71,248.8-84.81V0Z" fill="#ffffff"></path>
        </svg>
    </div>
</div>

<div class="container-fluid mt--7">
    <!-- Hero Section -->
    <div class="row mb-5">
        <div class="col-12">
            <div class="card shadow-lg border-0 overflow-hidden" style="background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);">
                <div class="card-body p-0">
                    <div class="row align-items-center min-vh-25">
                        <div class="col-12 p-5">
                            <div class="hero-content">
                                <div class="mb-4">
                                    <span class="badge badge-success badge-pill px-3 py-2 mb-3" style="font-size: 0.9rem;">
                                        <i class="fas fa-rocket mr-2"></i>
                                        WhatsApp Marketing Automation
                                    </span>
                                    <h2 class="display-5 font-weight-bold text-dark mb-3 hero-title">
                                        Transform Your Contact Management
                                    </h2>
                                    <p class="lead text-muted mb-4" style="font-size: 1.1rem; line-height: 1.6;">
                                        Intelligent import system that automatically normalizes phone numbers, 
                                        integrates with CRMs, and creates smart campaigns respecting WhatsApp API limits.
                                    </p>
                                </div>
                                
                                <!-- Feature Cards -->
                                <div class="row g-4">
                                    <div class="col-lg-4 col-md-12">
                                        <div class="feature-card-large h-100 p-4 rounded-lg bg-white shadow-sm border-0 hover-lift">
                                            <div class="d-flex align-items-center mb-3">
                                                <div class="feature-icon-large mr-4">
                                                    <div class="icon-circle-large bg-success-light">
                                                        <i class="fas fa-mobile-alt text-success fa-2x"></i>
                                                    </div>
                                                </div>
                                                <h4 class="font-weight-bold text-dark mb-0">Smart Phone Normalization</h4>
                                            </div>
                                            <p class="text-muted mb-0" style="font-size: 1rem;">Auto-format any phone number format to WhatsApp standard</p>
                                        </div>
                                    </div>
                                    <div class="col-lg-4 col-md-12">
                                        <div class="feature-card-large h-100 p-4 rounded-lg bg-white shadow-sm border-0 hover-lift">
                                            <div class="d-flex align-items-center mb-3">
                                                <div class="feature-icon-large mr-4">
                                                    <div class="icon-circle-large bg-primary-light">
                                                        <i class="fas fa-cloud-download-alt text-primary fa-2x"></i>
                                                    </div>
                                                </div>
                                                <h4 class="font-weight-bold text-dark mb-0">CRM Integration</h4>
                                            </div>
                                            <p class="text-muted mb-0" style="font-size: 1rem;">Direct import from HubSpot, Salesforce, and more</p>
                                        </div>
                                    </div>
                                    <div class="col-lg-4 col-md-12">
                                        <div class="feature-card-large h-100 p-4 rounded-lg bg-white shadow-sm border-0 hover-lift">
                                            <div class="d-flex align-items-center mb-3">
                                                <div class="feature-icon-large mr-4">
                                                    <div class="icon-circle-large bg-warning-light">
                                                        <i class="fas fa-chart-line text-warning fa-2x"></i>
                                                    </div>
                                                </div>
                                                <h4 class="font-weight-bold text-dark mb-0">Smart Campaigns</h4>
                                            </div>
                                            <p class="text-muted mb-0" style="font-size: 1rem;">Intelligent batching with API rate limits</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Import Methods -->
    <div class="row">
        <div class="col-12">
            <div class="card shadow-lg border-0">
                <div class="card-header bg-white border-0 pb-0">
                    <div class="row align-items-center">
                        <div class="col-md-8">
                            <h3 class="mb-0 text-dark">
                                <i class="fas fa-upload text-success mr-2"></i>
                                Import Your Contacts
                            </h3>
                            <p class="text-muted mb-0">Choose your preferred import method</p>
                        </div>
                        <div class="col-md-4 text-right">
                            <button type="button" class="btn btn-success btn-lg" id="downloadDemoBtn" onclick="window.location.href='{{ route('contacts.import.demo-csv') }}'">
                                <i class="fas fa-download mr-2"></i>
                                Download Sample
                            </button>
                        </div>
                    </div>
                </div>

                <div class="col-12">
                    @include('partials.flash')
                </div>

                <div class="card-body p-4">
                    <!-- Import Methods Tabs -->
                    <ul class="nav nav-pills nav-fill mb-4" id="importTabs" role="tablist">
                        <li class="nav-item">
                            <a class="nav-link active" id="csv-tab" data-toggle="tab" href="#csv-import" role="tab">
                                <i class="fas fa-file-csv mr-2"></i>
                                <span class="d-none d-md-inline">CSV Import</span>
                            </a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" id="crm-tab" data-toggle="tab" href="#crm-import" role="tab">
                                <i class="fas fa-cloud mr-2"></i>
                                <span class="d-none d-md-inline">CRM Integration</span>
                            </a>
                        </li>
                    </ul>

                    <div class="tab-content" id="importTabsContent">
                        <!-- CSV Import Tab -->
                        <div class="tab-pane fade show active" id="csv-import" role="tabpanel">
                            <!-- CSV Import Guidelines - Amazing UI -->
                            <div class="csv-guidelines-spectacular mb-5">
                                <!-- Pure Animation Background -->
                                <div class="position-absolute w-100 h-100" style="top: 0; left: 0; z-index: 1;">
                                    <!-- Floating Geometric Shapes -->
                                    <div class="animated-shape shape-circle-1"></div>
                                    <div class="animated-shape shape-circle-2"></div>
                                    <div class="animated-shape shape-triangle-1"></div>
                                    <div class="animated-shape shape-square-1"></div>
                                    <div class="animated-shape shape-hexagon-1"></div>
                                    <div class="animated-shape shape-diamond-1"></div>
                                    
                                    <!-- Floating Particles -->
                                    <div class="particle particle-1"></div>
                                    <div class="particle particle-2"></div>
                                    <div class="particle particle-3"></div>
                                    <div class="particle particle-4"></div>
                                    <div class="particle particle-5"></div>
                                    <div class="particle particle-6"></div>
                                    <div class="particle particle-7"></div>
                                    <div class="particle particle-8"></div>
                                    
                                    <!-- Animated Lines -->
                                    <div class="animated-line line-1"></div>
                                    <div class="animated-line line-2"></div>
                                    <div class="animated-line line-3"></div>
                                    
                                    <!-- Pulsing Dots -->
                                    <div class="pulsing-dot dot-1"></div>
                                    <div class="pulsing-dot dot-2"></div>
                                    <div class="pulsing-dot dot-3"></div>
                                    <div class="pulsing-dot dot-4"></div>
                                    <div class="pulsing-dot dot-5"></div>
                                    
                                    <!-- Rotating Elements -->
                                    <div class="rotating-element rotate-1"></div>
                                    <div class="rotating-element rotate-2"></div>
                                    <div class="rotating-element rotate-3"></div>
                                </div>

                                <!-- Header with Gradient -->
                                <div class="text-center mb-5 position-relative">
                                    <div class="guidelines-header-badge mb-4">
                                        <span class="badge-spectacular">
                                            <i class="fas fa-file-csv mr-2"></i>
                                            CSV Import Made Easy
                                        </span>
                                    </div>
                                    <h2 class="guidelines-title mb-4">
                                        <span class="title-gradient">Import Guidelines</span>
                                        <div class="title-underline"></div>
                                    </h2>
                                    <p class="guidelines-subtitle">
                                        Everything you need to know for <strong>perfect contact imports</strong>
                                    </p>
                                </div>

                                <!-- Spectacular Cards -->
                                <div class="row g-5">
                                    <!-- Phone Numbers Card - Enhanced -->
                                    <div class="col-lg-6">
                                        <div class="spectacular-card phone-card">
                                            <div class="card-glow-effect"></div>
                                            <div class="card-header-spectacular">
                                                <div class="card-icon-spectacular phone-icon">
                                                    <i class="fas fa-mobile-alt"></i>
                                                    <div class="icon-pulse"></div>
                                                </div>
                                                <div class="card-title-section">
                                                    <h4 class="card-title-spectacular">📱 Phone Numbers</h4>
                                                    <p class="card-subtitle-spectacular">Smart Auto-Formatting</p>
                                                </div>
                                            </div>
                                            
                                            <div class="card-content-spectacular">
                                                <div class="feature-highlight mb-4">
                                                    <i class="fas fa-magic text-success mr-2"></i>
                                                    <span>Don't worry about formatting! We handle any phone number style</span>
                                                </div>
                                                
                                                <div class="phone-transformation-showcase">
                                                    <div class="transformation-item">
                                                        <div class="phone-before-spectacular">+971 55 869 4287</div>
                                                        <div class="transformation-arrow">
                                                            <i class="fas fa-arrow-right"></i>
                                                            <div class="arrow-glow"></div>
                                                        </div>
                                                        <div class="phone-after-spectacular">+971558694287</div>
                                                    </div>
                                                    <div class="transformation-item">
                                                        <div class="phone-before-spectacular">971558694287</div>
                                                        <div class="transformation-arrow">
                                                            <i class="fas fa-arrow-right"></i>
                                                            <div class="arrow-glow"></div>
                                                        </div>
                                                        <div class="phone-after-spectacular">+971558694287</div>
                                                    </div>
                                                    <div class="transformation-item">
                                                        <div class="phone-before-spectacular">+1 (555) 123-4567</div>
                                                        <div class="transformation-arrow">
                                                            <i class="fas fa-arrow-right"></i>
                                                            <div class="arrow-glow"></div>
                                                        </div>
                                                        <div class="phone-after-spectacular">+15551234567</div>
                                                    </div>
                                                </div>

                                                <div class="card-footer-spectacular">
                                                    <div class="success-badge">
                                                        <i class="fas fa-check-circle mr-2"></i>
                                                        Automatic WhatsApp Formatting
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- CSV Columns Card - Enhanced -->
                                    <div class="col-lg-6">
                                        <div class="spectacular-card columns-card">
                                            <div class="card-glow-effect"></div>
                                            <div class="card-header-spectacular">
                                                <div class="card-icon-spectacular columns-icon">
                                                    <i class="fas fa-table"></i>
                                                    <div class="icon-pulse"></div>
                                                </div>
                                                <div class="card-title-section">
                                                    <h4 class="card-title-spectacular">📋 CSV Columns</h4>
                                                    <p class="card-subtitle-spectacular">Required Structure</p>
                                                </div>
                                            </div>
                                            
                                            <div class="card-content-spectacular">
                                                <div class="feature-highlight mb-4">
                                                    <i class="fas fa-info-circle text-primary mr-2"></i>
                                                    <span>Your CSV needs these column headers</span>
                                                </div>
                                                
                                                <!-- Required Fields - Enhanced -->
                                                <div class="field-section-spectacular mb-4">
                                                    <div class="section-header-spectacular">
                                                        <span class="badge-required">
                                                            <i class="fas fa-exclamation-circle mr-1"></i>
                                                            Required
                                                        </span>
                                                    </div>
                                                    <div class="fields-container-spectacular">
                                                        <div class="field-spectacular required-field">
                                                            <code class="field-code-spectacular">name</code>
                                                            <span class="field-description">Contact's full name</span>
                                                            <div class="field-glow"></div>
                                                        </div>
                                                        <div class="field-spectacular required-field">
                                                            <code class="field-code-spectacular">phone</code>
                                                            <span class="field-description">Any phone format</span>
                                                            <div class="field-glow"></div>
                                                        </div>
                                                    </div>
                                                </div>

                                                <!-- Optional Fields - Enhanced -->
                                                <div class="field-section-spectacular">
                                                    <div class="section-header-spectacular">
                                                        <span class="badge-optional">
                                                            <i class="fas fa-plus-circle mr-1"></i>
                                                            Optional
                                                        </span>
                                                    </div>
                                                    <div class="fields-container-spectacular">
                                                        <div class="field-spectacular optional-field">
                                                            <code class="field-code-spectacular">email</code>
                                                            <span class="field-description">Email address</span>
                                                            <div class="field-glow"></div>
                                                        </div>
                                                        <div class="field-spectacular optional-field">
                                                            <code class="field-code-spectacular">avatar</code>
                                                            <span class="field-description">Profile picture URL</span>
                                                            <div class="field-glow"></div>
                                                        </div>
                                                    </div>
                                                </div>

                                                <div class="card-footer-spectacular">
                                                    <div class="info-badge">
                                                        <i class="fas fa-lightbulb mr-2"></i>
                                                        Add any custom columns you need
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Import Form -->
                            <div class="card border-0 shadow-sm">
                                <div class="card-header bg-light border-0">
                                    <h6 class="mb-0 text-dark">
                                        <i class="fas fa-upload mr-2"></i>
                                        Upload Your CSV File
                                    </h6>
                                </div>
                                <div class="card-body">
                                    <form action="{{ route('contacts.import.store') }}" method="POST" enctype="multipart/form-data" id="importForm">
                                        @csrf

                                        <div class="row">
                                            <div class="col-md-6">
                                                <div class="form-group">
                                                    <label for="csv" class="form-label">
                                                        <i class="fas fa-file-csv mr-2"></i>
                                                        CSV File
                                                    </label>
                                                    <div class="custom-file">
                                                        <input type="file" class="custom-file-input" id="csv" name="csv" accept=".csv" required>
                                                        <label class="custom-file-label" for="csv" id="csv-label">Choose CSV file...</label>
                                                    </div>
                                                    <small class="form-text text-muted">
                                                        <i class="fas fa-info-circle mr-1"></i>
                                                        Upload your CSV file with the correct format
                                                    </small>
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="form-group">
                                                    <label for="group" class="form-label">
                                                        <i class="fas fa-users mr-2"></i>
                                                        Group Assignment
                                                    </label>
                                                    <select class="form-control" id="group" name="group">
                                                        <option value="">Select a group (optional)</option>
                                                        @foreach($groups as $id => $name)
                                                        <option value="{{ $id }}">{{ $name }}</option>
                                                        @endforeach
                                                    </select>
                                                    <small class="form-text text-muted">
                                                        <i class="fas fa-info-circle mr-1"></i>
                                                        Assign imported contacts to a group
                                                    </small>
                                                </div>
                                            </div>
                                        </div>

                                        <div class="form-group mt-4">
                                            <button type="submit" class="btn btn-success btn-lg mr-3" id="importBtn">
                                                <i class="fas fa-upload mr-2"></i>
                                                Import Contacts
                                            </button>
                                            <button type="button" class="btn btn-outline-primary btn-lg" id="previewCsvBtn">
                                                <i class="fas fa-eye mr-2"></i>
                                                Preview CSV
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>

                        <!-- CRM Import Tab -->
                        <div class="tab-pane fade" id="crm-import" role="tabpanel">
                            <div class="mt-4">
                                @if(isset($crmConnections) && $crmConnections->isNotEmpty())
                                    <div class="row">
                                        @foreach($crmConnections as $connection)
                                            <div class="col-md-6 col-lg-4 mb-4">
                                                <div class="card h-100">
                                                    <div class="card-body">
                                                        <div class="d-flex justify-content-between align-items-start mb-3">
                                                            <h5 class="card-title">{{ $connection->name }}</h5>
                                                            <span class="badge badge-{{ $connection->is_healthy ? 'success' : 'danger' }}">
                                                                {{ $connection->is_healthy ? __('crm.healthy') : __('crm.unhealthy') }}
                                                            </span>
                                                        </div>
                                                        <p class="card-text">
                                                            <strong>{{ __('crm.type') }}:</strong> {{ $connection->type_display }}<br>
                                                            <strong>{{ __('crm.last_sync') }}:</strong> 
                                                            @if($connection->last_sync_at)
                                                                {{ $connection->last_sync_at->diffForHumans() }}
                                                            @else
                                                                {{ __('crm.never') }}
                                                            @endif
                                                        </p>
                                                        <a href="{{ route('contacts.crm-import.index') }}?connection={{ $connection->id }}" 
                                                           class="btn btn-primary btn-sm">
                                                            <i class="fas fa-download"></i> {{ __('crm.import_from_crm') }}
                                                        </a>
                                                    </div>
                                                </div>
                                            </div>
                                        @endforeach
                                    </div>
                                    
                                    <div class="text-center mt-4">
                                        <a href="{{ route('contacts.crm-dashboard.index') }}" class="btn btn-outline-primary">
                                            <i class="fas fa-cog"></i> {{ __('crm.advanced_crm_import') }}
                                        </a>
                                    </div>
                                @else
                                    <div class="text-center py-5">
                                        <i class="fas fa-plug fa-3x text-muted mb-3"></i>
                                        <h4>{{ __('crm.no_crm_connections') }}</h4>
                                        <p class="text-muted">{{ __('crm.create_crm_connection_first') }}</p>
                                        <a href="{{ route('contacts.crm.create') }}" class="btn btn-primary">
                                            <i class="fas fa-plus"></i> {{ __('crm.create_connection') }}
                                        </a>
                                    </div>
                                @endif
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- CSV Preview Modal -->
<div class="modal fade" id="csvPreviewModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-xl" role="document" style="max-width: 95vw;">
        <div class="modal-content border-0 shadow-lg">
            <div class="modal-header bg-success text-white border-0">
                <h5 class="modal-title">
                    <i class="fas fa-eye mr-2"></i>
                    CSV Preview
                </h5>
                <button type="button" class="close text-white" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <div class="modal-body p-0">
                <div id="csvPreviewContent" style="max-height: 70vh; overflow: auto;">
                    <div class="text-center text-muted py-5">
                        <i class="fas fa-file-csv fa-3x mb-3"></i>
                        <p>Upload a CSV file to preview its contents.</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

@include('layouts.footers.auth')
@endsection

<script>
// Simple, direct JavaScript - no dependencies
console.log('CSV Import: Script starting...');

// Wait for page to load completely
window.addEventListener('load', function() {
    console.log('CSV Import: Page loaded, initializing...');
    
    // Test if elements exist
    const fileInput = document.getElementById('csv');
    const fileLabel = document.getElementById('csv-label');
    const previewBtn = document.getElementById('previewCsvBtn');
    
    console.log('CSV Import: Elements found:', {
        fileInput: !!fileInput,
        fileLabel: !!fileLabel, 
        previewBtn: !!previewBtn
    });
    
    if (!fileInput || !fileLabel || !previewBtn) {
        console.error('CSV Import: Required elements not found!');
        return;
    }
    
    // File input handler - SIMPLE VERSION
    fileInput.addEventListener('change', function(e) {
        console.log('CSV Import: File input changed');
        const file = e.target.files[0];
        
        if (file) {
            console.log('CSV Import: File selected:', file.name);
            fileLabel.textContent = file.name;
            fileLabel.setAttribute('data-content', file.name);
        } else {
            console.log('CSV Import: No file selected');
            fileLabel.textContent = 'Choose CSV file...';
            fileLabel.setAttribute('data-content', 'Choose CSV file...');
        }
    });
    
    // Preview button handler - SIMPLE VERSION
    previewBtn.addEventListener('click', function(e) {
        e.preventDefault();
        console.log('CSV Import: Preview button clicked');
        
        const file = fileInput.files[0];
        if (!file) {
            alert('Please select a CSV file first.');
            return;
        }
        
        console.log('CSV Import: Processing file:', file.name);
        
        // Show loading
        const originalText = previewBtn.innerHTML;
        previewBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Loading...';
        previewBtn.disabled = true;
        
        // Read file
        const reader = new FileReader();
        reader.onload = function(e) {
            try {
                const csv = e.target.result;
                console.log('CSV Import: File read successfully, length:', csv.length);
                
                // Simple CSV parsing
                const lines = csv.split('\n').filter(line => line.trim());
                if (lines.length === 0) {
                    alert('CSV file is empty.');
                    return;
                }
                
                // Get first few lines for preview
                const headers = lines[0].split(',');
                const dataRows = lines.slice(1, 6);
                
                console.log('CSV Import: Headers:', headers);
                console.log('CSV Import: Data rows:', dataRows.length);
                
                // Build simple table with responsive design
                let html = `
                    <div class="table-responsive" style="max-height: 400px; overflow: auto;">
                        <table class="table table-striped table-hover table-sm">
                            <thead class="thead-dark sticky-top">
                                <tr>`;
                
                headers.forEach(header => {
                    html += `<th class="text-nowrap px-3 py-2" style="min-width: 120px;">${header.trim()}</th>`;
                });
                
                html += `</tr></thead><tbody>`;
                
                dataRows.forEach(row => {
                    const cells = row.split(',');
                    html += '<tr>';
                    headers.forEach((header, index) => {
                        const cellValue = cells[index] || '';
                        html += `<td class="text-nowrap px-3 py-2" style="max-width: 200px; overflow: hidden; text-overflow: ellipsis;" title="${cellValue.trim()}">${cellValue.trim()}</td>`;
                    });
                    html += '</tr>';
                });
                
                html += `</tbody></table>
                    </div>
                    <div class="bg-light p-3 border-top">
                        <div class="row text-center">
                            <div class="col-md-3">
                                <h6 class="mb-1 text-success">${lines.length - 1}</h6>
                                <small class="text-muted">Total Rows</small>
                            </div>
                            <div class="col-md-3">
                                <h6 class="mb-1 text-primary">${headers.length}</h6>
                                <small class="text-muted">Columns</small>
                            </div>
                            <div class="col-md-3">
                                <h6 class="mb-1 text-info">${(file.size / 1024).toFixed(2)} KB</h6>
                                <small class="text-muted">File Size</small>
                            </div>
                            <div class="col-md-3">
                                <h6 class="mb-1 text-warning">${Math.min(5, dataRows.length)}</h6>
                                <small class="text-muted">Preview Rows</small>
                            </div>
                        </div>
                        <div class="text-center mt-2">
                            <small class="text-muted">
                                <i class="fas fa-info-circle mr-1"></i>
                                Scroll horizontally to view all columns. Hover over cells to see full content.
                            </small>
                        </div>
                    </div>`;
                
                // Update modal content
                const modalContent = document.getElementById('csvPreviewContent');
                if (modalContent) {
                    modalContent.innerHTML = html;
                    console.log('CSV Import: Modal content updated');
                }
                
                // Show modal - try simple method first
                const modal = document.getElementById('csvPreviewModal');
                if (modal) {
                    modal.style.display = 'block';
                    modal.classList.add('show');
                    document.body.classList.add('modal-open');
                    
                    // Add backdrop
                    const backdrop = document.createElement('div');
                    backdrop.className = 'modal-backdrop fade show';
                    backdrop.onclick = function() {
                        modal.style.display = 'none';
                        modal.classList.remove('show');
                        document.body.classList.remove('modal-open');
                        backdrop.remove();
                    };
                    document.body.appendChild(backdrop);
                    
                    // Add close button handler
                    const closeBtn = modal.querySelector('.close');
                    if (closeBtn) {
                        closeBtn.onclick = function() {
                            modal.style.display = 'none';
                            modal.classList.remove('show');
                            document.body.classList.remove('modal-open');
                            backdrop.remove();
                        };
                    }
                    
                    console.log('CSV Import: Modal shown');
                } else {
                    alert('Preview ready but modal not found. Check console.');
                }
                
            } catch (error) {
                console.error('CSV Import: Error processing file:', error);
                alert('Error processing CSV: ' + error.message);
            } finally {
                // Restore button
                previewBtn.innerHTML = originalText;
                previewBtn.disabled = false;
            }
        };
        
        reader.onerror = function() {
            console.error('CSV Import: Error reading file');
            alert('Error reading file. Please try again.');
            previewBtn.innerHTML = originalText;
            previewBtn.disabled = false;
        };
        
        reader.readAsText(file);
    });
    
    console.log('CSV Import: All handlers attached successfully');
});

// Also try with a shorter delay
setTimeout(function() {
    console.log('CSV Import: Timeout initialization...');
    
    const fileInput = document.getElementById('csv');
    const fileLabel = document.getElementById('csv-label');
    
    if (fileInput && fileLabel) {
        console.log('CSV Import: Elements found in timeout, adding backup handler');
        
        fileInput.onchange = function() {
            console.log('CSV Import: Backup file handler triggered');
            const file = this.files[0];
            if (file) {
                fileLabel.textContent = file.name;
                console.log('CSV Import: Backup handler updated label to:', file.name);
            }
        };
    }
}, 1000);

console.log('CSV Import: Script loaded completely');
</script>

@push('styles')
<style>
    /* WhatsApp Brand Colors */
    :root {
        --whatsapp-green: #25D366;
        --whatsapp-dark-green: #128C7E;
        --whatsapp-blue: #34B7F1;
        --whatsapp-dark-blue: #075E54;
    }

    /* Enhanced Hero Section */
    .hero-title {
        background: linear-gradient(135deg, var(--whatsapp-green) 0%, var(--whatsapp-dark-green) 100%);
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        background-clip: text;
    }

    .hero-content {
        animation: fadeInUp 0.8s ease-out;
    }

    /* Enhanced Header Section */
    .header {
        position: relative;
        overflow: hidden;
    }

    .header-content {
        animation: fadeInUp 1s ease-out;
    }

    .header-title {
        text-shadow: 0 2px 4px rgba(0,0,0,0.1);
        animation: slideInDown 1s ease-out;
    }

    .header-subtitle {
        text-shadow: 0 1px 2px rgba(0,0,0,0.1);
        animation: fadeInUp 1s ease-out 0.3s both;
    }

    .header-badge {
        animation: bounceIn 1s ease-out 0.5s both;
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    }

    .whatsapp-icon {
        animation: pulse 2s infinite;
        filter: drop-shadow(0 2px 4px rgba(0,0,0,0.2));
    }

    /* Header Features */
    .header-feature {
        padding: 1.5rem;
        border-radius: 15px;
        background: rgba(255,255,255,0.1);
        backdrop-filter: blur(10px);
        border: 1px solid rgba(255,255,255,0.2);
        transition: all 0.3s ease;
        animation: fadeInUp 1s ease-out 0.6s both;
    }

    .header-feature:hover {
        background: rgba(255,255,255,0.2);
        transform: translateY(-5px);
        box-shadow: 0 10px 25px rgba(0,0,0,0.15);
    }

    .header-feature-icon {
        width: 50px;
        height: 50px;
        border-radius: 50%;
        background: rgba(255,255,255,0.2);
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto;
        font-size: 1.2rem;
    }

    /* Floating Background Elements */
    .floating-element {
        position: absolute;
        border-radius: 50%;
        background: rgba(255,255,255,0.1);
        animation: float 6s ease-in-out infinite;
    }

    .floating-element-1 {
        width: 80px;
        height: 80px;
        top: 10%;
        left: 10%;
        animation-delay: 0s;
    }

    .floating-element-2 {
        width: 120px;
        height: 120px;
        top: 20%;
        right: 15%;
        animation-delay: 2s;
    }

    .floating-element-3 {
        width: 60px;
        height: 60px;
        bottom: 30%;
        left: 20%;
        animation-delay: 4s;
    }

    .floating-element-4 {
        width: 100px;
        height: 100px;
        bottom: 20%;
        right: 10%;
        animation-delay: 1s;
    }

    @keyframes float {
        0%, 100% {
            transform: translateY(0px) rotate(0deg);
            opacity: 0.7;
        }
        50% {
            transform: translateY(-20px) rotate(180deg);
            opacity: 1;
        }
    }

    /* Header Wave */
    .header-wave {
        position: absolute;
        bottom: 0;
        left: 0;
        width: 100%;
        overflow: hidden;
        line-height: 0;
    }

    .header-wave svg {
        position: relative;
        display: block;
        width: calc(100% + 1.3px);
        height: 60px;
    }

    /* Enhanced Animations */
    @keyframes slideInDown {
        from {
            opacity: 0;
            transform: translateY(-50px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    @keyframes bounceIn {
        0% {
            opacity: 0;
            transform: scale(0.3);
        }
        50% {
            opacity: 1;
            transform: scale(1.05);
        }
        70% {
            transform: scale(0.9);
        }
        100% {
            opacity: 1;
            transform: scale(1);
        }
    }

    /* Responsive Header */
    @media (max-width: 768px) {
        .header-title {
            font-size: 2.5rem;
        }
        
        .header-subtitle {
            font-size: 1.1rem;
        }
        
        .header-feature {
            padding: 1rem;
            margin-bottom: 1rem;
        }
        
        .floating-element {
            display: none;
        }
        
        .header-wave svg {
            height: 40px;
        }
    }

    /* Utility Classes */
    .min-vh-30 {
        min-height: 30vh;
    }

    /* Feature Cards */
    .feature-card {
        transition: all 0.3s ease;
        border: 1px solid rgba(0,0,0,0.05);
    }

    .feature-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 15px 35px rgba(0,0,0,0.1) !important;
    }

    /* Large Feature Cards */
    .feature-card-large {
        transition: all 0.3s ease;
        border: 1px solid rgba(0,0,0,0.05);
    }

    .feature-card-large:hover {
        transform: translateY(-3px);
        box-shadow: 0 10px 25px rgba(0,0,0,0.15) !important;
    }

    .hover-lift {
        transition: transform 0.3s ease, box-shadow 0.3s ease;
    }

    .hover-lift:hover {
        transform: translateY(-5px);
    }

    /* Icon Circles */
    .icon-circle {
        width: 50px;
        height: 50px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto;
    }

    /* Large Icon Circles */
    .icon-circle-large {
        width: 70px;
        height: 70px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        flex-shrink: 0;
    }

    .bg-success-light {
        background-color: rgba(40, 167, 69, 0.1);
    }

    .bg-primary-light {
        background-color: rgba(0, 123, 255, 0.1);
    }

    .bg-warning-light {
        background-color: rgba(255, 193, 7, 0.1);
    }

    /* WhatsApp Animation */
    .pulse-animation {
        animation: pulse 2s infinite;
    }

    @keyframes pulse {
        0% {
            transform: scale(1);
            opacity: 1;
        }
        50% {
            transform: scale(1.1);
            opacity: 0.8;
        }
        100% {
            transform: scale(1);
            opacity: 1;
        }
    }

    .whatsapp-animation {
        position: relative;
    }

    .whatsapp-animation::before {
        content: '';
        position: absolute;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        width: 120px;
        height: 120px;
        border: 2px solid rgba(255,255,255,0.3);
        border-radius: 50%;
        animation: ripple 2s infinite;
    }

    @keyframes ripple {
        0% {
            transform: translate(-50%, -50%) scale(0.8);
            opacity: 1;
        }
        100% {
            transform: translate(-50%, -50%) scale(1.5);
            opacity: 0;
        }
    }

    /* Stats Grid */
    .stat-item {
        padding: 1rem;
        border-radius: 10px;
        background: rgba(255,255,255,0.1);
        backdrop-filter: blur(10px);
        transition: all 0.3s ease;
    }

    .stat-item:hover {
        background: rgba(255,255,255,0.2);
        transform: scale(1.05);
    }

    /* Custom Animations */
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(30px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }

    @keyframes slideInRight {
        from {
            opacity: 0;
            transform: translateX(30px);
        }
        to {
            opacity: 1;
            transform: translateX(0);
        }
    }

    /* Responsive Design */
    @media (max-width: 768px) {
        .hero-title {
            font-size: 2rem;
        }
        
        .feature-card {
            margin-bottom: 1rem;
        }
        
        .hero-visual {
            min-height: 300px !important;
        }
        
        .stats-grid .col-6 {
            margin-bottom: 1rem;
        }
    }

    /* Card Styles */
    .card {
        animation: fadeInUp 0.6s ease-out;
        transition: all 0.3s ease;
    }

    .card:hover {
        transform: translateY(-5px);
        box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1) !important;
    }

    /* Button Styles */
    .btn {
        border-radius: 50px;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        transition: all 0.3s ease;
    }

    .btn:hover {
        transform: translateY(-2px);
        box-shadow: 0 10px 20px rgba(0, 0, 0, 0.2);
    }

    .btn-success {
        background: linear-gradient(135deg, var(--whatsapp-green) 0%, var(--whatsapp-dark-green) 100%);
        border: none;
    }

    .btn-success:hover {
        background: linear-gradient(135deg, var(--whatsapp-dark-green) 0%, var(--whatsapp-green) 100%);
    }

    /* Nav Pills */
    .nav-pills .nav-link {
        border-radius: 50px;
        margin: 0 5px;
        transition: all 0.3s ease;
    }

    .nav-pills .nav-link.active {
        background: linear-gradient(135deg, var(--whatsapp-green) 0%, var(--whatsapp-dark-green) 100%);
        color: white;
        transform: translateY(-2px);
        box-shadow: 0 5px 15px rgba(37, 211, 102, 0.4);
    }

    .nav-pills .nav-link:hover:not(.active) {
        background-color: rgba(37, 211, 102, 0.1);
        color: var(--whatsapp-green);
    }

    /* Phone Examples */
    .phone-examples {
        background: rgba(255, 255, 255, 0.1);
        border-radius: 10px;
        padding: 15px;
    }

    .phone-example {
        margin: 8px 0;
        font-family: 'Courier New', monospace;
    }

    .phone-example code {
        background: rgba(255, 255, 255, 0.2);
        color: white;
        padding: 5px 10px;
        border-radius: 5px;
        font-size: 0.9rem;
    }

    /* New Phone Examples Styling */
    .phone-examples-new {
        background: rgba(255, 255, 255, 0.8);
        border-radius: 10px;
        padding: 15px;
        border: 1px solid rgba(0,0,0,0.1);
    }

    .phone-example-new {
        font-family: 'Courier New', monospace;
        display: flex;
        align-items: center;
        padding: 8px 0;
    }

    .input-example {
        background: #f8f9fa;
        color: #495057;
        padding: 6px 12px;
        border-radius: 6px;
        font-size: 0.9rem;
        border: 1px solid #dee2e6;
        font-weight: 500;
    }

    .output-example {
        background: #d4edda;
        color: #155724;
        padding: 6px 12px;
        border-radius: 6px;
        font-size: 0.9rem;
        border: 1px solid #c3e6cb;
        font-weight: 600;
    }

    /* TEST CSS - REMOVE AFTER VERIFICATION */
    .csv-guidelines-spectacular {
        background: red !important;
        border: 5px solid blue !important;
    }
    
    /* SPECTACULAR CSV GUIDELINES - AMAZING UI */
    .csv-guidelines-spectacular {
        position: relative;
        background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 50%, #f1f3f4 100%);
        padding: 4rem 2rem;
        border-radius: 25px;
        overflow: hidden;
        box-shadow: 0 20px 60px rgba(0,0,0,0.1);
    }

    /* PURE ANIMATION BACKGROUND - NO TEXT */
    
    /* Animated Geometric Shapes */
    .animated-shape {
        position: absolute;
        pointer-events: none;
        opacity: 0.1;
    }

    .shape-circle-1 {
        width: 120px;
        height: 120px;
        background: linear-gradient(45deg, #25D366, #128C7E);
        border-radius: 50%;
        top: 15%;
        left: 10%;
        animation: floatRotate 12s ease-in-out infinite;
        animation-delay: 0s;
    }

    .shape-circle-2 {
        width: 80px;
        height: 80px;
        background: linear-gradient(135deg, #3498db, #2980b9);
        border-radius: 50%;
        top: 60%;
        right: 15%;
        animation: floatRotate 15s ease-in-out infinite reverse;
        animation-delay: 2s;
    }

    .shape-triangle-1 {
        width: 0;
        height: 0;
        border-left: 40px solid transparent;
        border-right: 40px solid transparent;
        border-bottom: 70px solid rgba(52, 152, 219, 0.3);
        top: 25%;
        right: 25%;
        animation: triangleFloat 10s ease-in-out infinite;
        animation-delay: 1s;
    }

    .shape-square-1 {
        width: 60px;
        height: 60px;
        background: linear-gradient(45deg, #e74c3c, #c0392b);
        top: 70%;
        left: 20%;
        animation: squareRotate 8s linear infinite;
        animation-delay: 3s;
        transform-origin: center;
    }

    .shape-hexagon-1 {
        width: 100px;
        height: 100px;
        background: linear-gradient(60deg, #f39c12, #e67e22);
        clip-path: polygon(50% 0%, 100% 25%, 100% 75%, 50% 100%, 0% 75%, 0% 25%);
        top: 40%;
        left: 5%;
        animation: hexagonPulse 6s ease-in-out infinite;
        animation-delay: 4s;
    }

    .shape-diamond-1 {
        width: 70px;
        height: 70px;
        background: linear-gradient(45deg, #9b59b6, #8e44ad);
        transform: rotate(45deg);
        top: 10%;
        right: 5%;
        animation: diamondSpin 14s ease-in-out infinite;
        animation-delay: 1.5s;
    }

    /* Floating Particles */
    .particle {
        position: absolute;
        background: radial-gradient(circle, rgba(255,255,255,0.8) 0%, rgba(255,255,255,0.2) 100%);
        border-radius: 50%;
        pointer-events: none;
    }

    .particle-1 {
        width: 8px;
        height: 8px;
        top: 20%;
        left: 30%;
        animation: particleFloat1 7s ease-in-out infinite;
    }

    .particle-2 {
        width: 12px;
        height: 12px;
        top: 50%;
        left: 60%;
        animation: particleFloat2 9s ease-in-out infinite;
        animation-delay: 1s;
    }

    .particle-3 {
        width: 6px;
        height: 6px;
        top: 80%;
        left: 40%;
        animation: particleFloat3 11s ease-in-out infinite;
        animation-delay: 2s;
    }

    .particle-4 {
        width: 10px;
        height: 10px;
        top: 35%;
        right: 20%;
        animation: particleFloat1 8s ease-in-out infinite reverse;
        animation-delay: 0.5s;
    }

    .particle-5 {
        width: 14px;
        height: 14px;
        top: 65%;
        right: 35%;
        animation: particleFloat2 10s ease-in-out infinite;
        animation-delay: 3s;
    }

    .particle-6 {
        width: 7px;
        height: 7px;
        top: 15%;
        left: 70%;
        animation: particleFloat3 6s ease-in-out infinite;
        animation-delay: 1.5s;
    }

    .particle-7 {
        width: 9px;
        height: 9px;
        top: 45%;
        left: 15%;
        animation: particleFloat1 12s ease-in-out infinite;
        animation-delay: 4s;
    }

    .particle-8 {
        width: 11px;
        height: 11px;
        top: 75%;
        right: 10%;
        animation: particleFloat2 7s ease-in-out infinite reverse;
        animation-delay: 2.5s;
    }

    /* Animated Lines */
    .animated-line {
        position: absolute;
        background: linear-gradient(90deg, transparent, rgba(37, 211, 102, 0.3), transparent);
        pointer-events: none;
    }

    .line-1 {
        width: 200px;
        height: 2px;
        top: 30%;
        left: -200px;
        animation: lineMove1 8s ease-in-out infinite;
    }

    .line-2 {
        width: 150px;
        height: 1px;
        top: 60%;
        right: -150px;
        animation: lineMove2 10s ease-in-out infinite;
        animation-delay: 2s;
    }

    .line-3 {
        width: 100px;
        height: 3px;
        top: 80%;
        left: -100px;
        animation: lineMove1 12s ease-in-out infinite;
        animation-delay: 4s;
    }

    /* Pulsing Dots */
    .pulsing-dot {
        position: absolute;
        border-radius: 50%;
        pointer-events: none;
    }

    .dot-1 {
        width: 20px;
        height: 20px;
        background: radial-gradient(circle, rgba(37, 211, 102, 0.6) 0%, rgba(37, 211, 102, 0.1) 100%);
        top: 25%;
        left: 50%;
        animation: dotPulse 4s ease-in-out infinite;
    }

    .dot-2 {
        width: 16px;
        height: 16px;
        background: radial-gradient(circle, rgba(52, 152, 219, 0.6) 0%, rgba(52, 152, 219, 0.1) 100%);
        top: 55%;
        left: 25%;
        animation: dotPulse 5s ease-in-out infinite;
        animation-delay: 1s;
    }

    .dot-3 {
        width: 24px;
        height: 24px;
        background: radial-gradient(circle, rgba(155, 89, 182, 0.6) 0%, rgba(155, 89, 182, 0.1) 100%);
        top: 40%;
        right: 30%;
        animation: dotPulse 6s ease-in-out infinite;
        animation-delay: 2s;
    }

    .dot-4 {
        width: 18px;
        height: 18px;
        background: radial-gradient(circle, rgba(231, 76, 60, 0.6) 0%, rgba(231, 76, 60, 0.1) 100%);
        top: 70%;
        left: 60%;
        animation: dotPulse 4.5s ease-in-out infinite;
        animation-delay: 0.5s;
    }

    .dot-5 {
        width: 22px;
        height: 22px;
        background: radial-gradient(circle, rgba(243, 156, 18, 0.6) 0%, rgba(243, 156, 18, 0.1) 100%);
        top: 15%;
        right: 45%;
        animation: dotPulse 5.5s ease-in-out infinite;
        animation-delay: 3s;
    }

    /* Rotating Elements */
    .rotating-element {
        position: absolute;
        pointer-events: none;
        border: 2px solid rgba(255, 255, 255, 0.2);
        border-radius: 50%;
    }

    .rotate-1 {
        width: 150px;
        height: 150px;
        top: 20%;
        right: 20%;
        border-top-color: rgba(37, 211, 102, 0.4);
        animation: elementRotate 20s linear infinite;
    }

    .rotate-2 {
        width: 100px;
        height: 100px;
        top: 50%;
        left: 35%;
        border-right-color: rgba(52, 152, 219, 0.4);
        animation: elementRotate 15s linear infinite reverse;
        animation-delay: 2s;
    }

    .rotate-3 {
        width: 80px;
        height: 80px;
        top: 75%;
        right: 40%;
        border-bottom-color: rgba(155, 89, 182, 0.4);
        animation: elementRotate 25s linear infinite;
        animation-delay: 5s;
    }

    /* ANIMATION KEYFRAMES */
    
    @keyframes floatRotate {
        0%, 100% {
            transform: translateY(0px) rotate(0deg) scale(1);
            opacity: 0.1;
        }
        25% {
            transform: translateY(-30px) rotate(90deg) scale(1.1);
            opacity: 0.2;
        }
        50% {
            transform: translateY(-15px) rotate(180deg) scale(0.9);
            opacity: 0.15;
        }
        75% {
            transform: translateY(-25px) rotate(270deg) scale(1.05);
            opacity: 0.18;
        }
    }

    @keyframes triangleFloat {
        0%, 100% {
            transform: translateY(0px) rotate(0deg);
            opacity: 0.1;
        }
        33% {
            transform: translateY(-20px) rotate(120deg);
            opacity: 0.2;
        }
        66% {
            transform: translateY(10px) rotate(240deg);
            opacity: 0.15;
        }
    }

    @keyframes squareRotate {
        0% {
            transform: rotate(0deg) scale(1);
            opacity: 0.1;
        }
        25% {
            transform: rotate(90deg) scale(1.2);
            opacity: 0.2;
        }
        50% {
            transform: rotate(180deg) scale(0.8);
            opacity: 0.15;
        }
        75% {
            transform: rotate(270deg) scale(1.1);
            opacity: 0.18;
        }
        100% {
            transform: rotate(360deg) scale(1);
            opacity: 0.1;
        }
    }

    @keyframes hexagonPulse {
        0%, 100% {
            transform: scale(1) rotate(0deg);
            opacity: 0.1;
        }
        50% {
            transform: scale(1.3) rotate(180deg);
            opacity: 0.25;
        }
    }

    @keyframes diamondSpin {
        0% {
            transform: rotate(45deg) scale(1);
            opacity: 0.1;
        }
        25% {
            transform: rotate(135deg) scale(1.2);
            opacity: 0.2;
        }
        50% {
            transform: rotate(225deg) scale(0.9);
            opacity: 0.15;
        }
        75% {
            transform: rotate(315deg) scale(1.1);
            opacity: 0.18;
        }
        100% {
            transform: rotate(405deg) scale(1);
            opacity: 0.1;
        }
    }

    @keyframes particleFloat1 {
        0%, 100% {
            transform: translateY(0px) translateX(0px);
            opacity: 0.3;
        }
        25% {
            transform: translateY(-40px) translateX(20px);
            opacity: 0.8;
        }
        50% {
            transform: translateY(-20px) translateX(-15px);
            opacity: 0.5;
        }
        75% {
            transform: translateY(-35px) translateX(10px);
            opacity: 0.7;
        }
    }

    @keyframes particleFloat2 {
        0%, 100% {
            transform: translateY(0px) translateX(0px) scale(1);
            opacity: 0.4;
        }
        33% {
            transform: translateY(-25px) translateX(-20px) scale(1.2);
            opacity: 0.8;
        }
        66% {
            transform: translateY(-45px) translateX(15px) scale(0.8);
            opacity: 0.6;
        }
    }

    @keyframes particleFloat3 {
        0%, 100% {
            transform: translateY(0px) translateX(0px) rotate(0deg);
            opacity: 0.2;
        }
        50% {
            transform: translateY(-30px) translateX(25px) rotate(180deg);
            opacity: 0.7;
        }
    }

    @keyframes lineMove1 {
        0% {
            left: -200px;
            opacity: 0;
        }
        50% {
            opacity: 1;
        }
        100% {
            left: 100%;
            opacity: 0;
        }
    }

    @keyframes lineMove2 {
        0% {
            right: -150px;
            opacity: 0;
        }
        50% {
            opacity: 1;
        }
        100% {
            right: 100%;
            opacity: 0;
        }
    }

    @keyframes dotPulse {
        0%, 100% {
            transform: scale(1);
            opacity: 0.3;
        }
        50% {
            transform: scale(1.5);
            opacity: 0.8;
        }
    }

    @keyframes elementRotate {
        0% {
            transform: rotate(0deg);
            opacity: 0.2;
        }
        50% {
            opacity: 0.4;
        }
        100% {
            transform: rotate(360deg);
            opacity: 0.2;
        }
    }

    /* Responsive - Hide complex animations on mobile */
    @media (max-width: 768px) {
        .animated-shape,
        .animated-line,
        .rotating-element {
            display: none;
        }
        
        .particle {
            width: 6px !important;
            height: 6px !important;
        }
        
        .pulsing-dot {
            width: 12px !important;
            height: 12px !important;
        }
    }

    /* Header Styling */
    .guidelines-header-badge {
        position: relative;
        z-index: 2;
    }

    .badge-spectacular {
        background: linear-gradient(135deg, #25D366 0%, #128C7E 100%);
        color: white;
        padding: 1rem 2rem;
        border-radius: 50px;
        font-size: 1rem;
        font-weight: 600;
        letter-spacing: 0.5px;
        box-shadow: 0 10px 30px rgba(37, 211, 102, 0.3);
        animation: pulseGlow 3s ease-in-out infinite;
        display: inline-block;
    }

    @keyframes pulseGlow {
        0%, 100% {
            box-shadow: 0 10px 30px rgba(37, 211, 102, 0.3);
            transform: scale(1);
        }
        50% {
            box-shadow: 0 15px 40px rgba(37, 211, 102, 0.5);
            transform: scale(1.05);
        }
    }

    .guidelines-title {
        position: relative;
        z-index: 2;
    }

    .title-gradient {
        background: linear-gradient(135deg, #2c3e50 0%, #34495e 50%, #2c3e50 100%);
        -webkit-background-clip: text;
        -webkit-text-fill-color: transparent;
        background-clip: text;
        font-size: 3rem;
        font-weight: 800;
        letter-spacing: -1px;
        text-shadow: 0 4px 8px rgba(0,0,0,0.1);
    }

    .title-underline {
        width: 100px;
        height: 4px;
        background: linear-gradient(135deg, #25D366 0%, #128C7E 100%);
        margin: 1rem auto;
        border-radius: 2px;
        animation: expandLine 2s ease-out;
    }

    @keyframes expandLine {
        0% {
            width: 0;
        }
        100% {
            width: 100px;
        }
    }

    .guidelines-subtitle {
        font-size: 1.2rem;
        color: #6c757d;
        position: relative;
        z-index: 2;
        max-width: 600px;
        margin: 0 auto;
    }

    /* Spectacular Cards */
    .spectacular-card {
        background: rgba(255, 255, 255, 0.95);
        backdrop-filter: blur(20px);
        border: 1px solid rgba(255, 255, 255, 0.3);
        border-radius: 20px;
        padding: 2.5rem;
        position: relative;
        overflow: hidden;
        transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
        height: 100%;
        z-index: 2;
    }

    .spectacular-card:hover {
        transform: translateY(-10px) scale(1.02);
        box-shadow: 0 25px 60px rgba(0,0,0,0.15);
    }

    .card-glow-effect {
        position: absolute;
        top: -2px;
        left: -2px;
        right: -2px;
        bottom: -2px;
        background: linear-gradient(45deg, #25D366, #128C7E, #25D366);
        border-radius: 22px;
        opacity: 0;
        transition: opacity 0.3s ease;
        z-index: -1;
    }

    .spectacular-card:hover .card-glow-effect {
        opacity: 0.3;
        animation: rotateGlow 3s linear infinite;
    }

    @keyframes rotateGlow {
        0% {
            transform: rotate(0deg);
        }
        100% {
            transform: rotate(360deg);
        }
    }

    /* Card Headers */
    .card-header-spectacular {
        display: flex;
        align-items: center;
        margin-bottom: 2rem;
        padding-bottom: 1.5rem;
        border-bottom: 2px solid #f8f9fa;
    }

    .card-icon-spectacular {
        width: 70px;
        height: 70px;
        border-radius: 18px;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-right: 1.5rem;
        font-size: 1.8rem;
        position: relative;
        flex-shrink: 0;
    }

    .phone-icon {
        background: linear-gradient(135deg, #25D366 0%, #128C7E 100%);
        color: white;
    }

    .columns-icon {
        background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
        color: white;
    }

    .icon-pulse {
        position: absolute;
        top: -5px;
        left: -5px;
        right: -5px;
        bottom: -5px;
        border: 2px solid currentColor;
        border-radius: 23px;
        opacity: 0.3;
        animation: iconPulse 2s ease-in-out infinite;
    }

    @keyframes iconPulse {
        0%, 100% {
            transform: scale(1);
            opacity: 0.3;
        }
        50% {
            transform: scale(1.1);
            opacity: 0.1;
        }
    }

    .card-title-section {
        flex: 1;
    }

    .card-title-spectacular {
        font-size: 1.5rem;
        font-weight: 700;
        color: #2c3e50;
        margin-bottom: 0.5rem;
        letter-spacing: -0.5px;
    }

    .card-subtitle-spectacular {
        color: #6c757d;
        font-size: 0.9rem;
        font-weight: 500;
        margin: 0;
        text-transform: uppercase;
        letter-spacing: 1px;
    }

    /* Feature Highlights */
    .feature-highlight {
        background: linear-gradient(135deg, rgba(37, 211, 102, 0.1) 0%, rgba(18, 140, 126, 0.1) 100%);
        padding: 1rem 1.5rem;
        border-radius: 12px;
        border-left: 4px solid #25D366;
        font-weight: 500;
        color: #2c3e50;
    }

    /* Phone Transformation Showcase */
    .phone-transformation-showcase {
        background: #f8f9fa;
        border-radius: 15px;
        padding: 2rem;
        margin: 2rem 0;
        border: 1px solid #e9ecef;
    }

    .transformation-item {
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-bottom: 1.5rem;
        padding: 1.2rem;
        background: white;
        border-radius: 12px;
        box-shadow: 0 4px 15px rgba(0,0,0,0.05);
        transition: all 0.3s ease;
    }

    .transformation-item:hover {
        transform: translateX(5px);
        box-shadow: 0 8px 25px rgba(0,0,0,0.1);
    }

    .transformation-item:last-child {
        margin-bottom: 0;
    }

    .phone-before-spectacular {
        font-family: 'Courier New', monospace;
        background: linear-gradient(135deg, #e9ecef 0%, #dee2e6 100%);
        padding: 0.8rem 1.2rem;
        border-radius: 8px;
        color: #495057;
        font-size: 0.95rem;
        font-weight: 600;
        flex: 1;
        text-align: center;
        border: 2px solid #dee2e6;
    }

    .transformation-arrow {
        position: relative;
        color: #25D366;
        font-size: 1.2rem;
        margin: 0 1.5rem;
        flex-shrink: 0;
    }

    .arrow-glow {
        position: absolute;
        top: -10px;
        left: -10px;
        right: -10px;
        bottom: -10px;
        background: radial-gradient(circle, rgba(37, 211, 102, 0.2) 0%, transparent 70%);
        border-radius: 50%;
        animation: arrowGlow 2s ease-in-out infinite;
    }

    @keyframes arrowGlow {
        0%, 100% {
            opacity: 0.5;
            transform: scale(1);
        }
        50% {
            opacity: 1;
            transform: scale(1.2);
        }
    }

    .phone-after-spectacular {
        font-family: 'Courier New', monospace;
        background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
        padding: 0.8rem 1.2rem;
        border-radius: 8px;
        color: #155724;
        font-size: 0.95rem;
        font-weight: 700;
        flex: 1;
        text-align: center;
        border: 2px solid #c3e6cb;
        position: relative;
    }

    .phone-after-spectacular::after {
        content: '✓';
        position: absolute;
        top: -8px;
        right: -8px;
        background: #28a745;
        color: white;
        width: 20px;
        height: 20px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 0.7rem;
        font-weight: bold;
    }

    /* Field Sections */
    .field-section-spectacular {
        margin-bottom: 2rem;
    }

    .section-header-spectacular {
        margin-bottom: 1.5rem;
    }

    .badge-required {
        background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
        color: white;
        padding: 0.6rem 1.2rem;
        border-radius: 25px;
        font-size: 0.85rem;
        font-weight: 600;
        letter-spacing: 0.5px;
        box-shadow: 0 4px 15px rgba(220, 53, 69, 0.3);
    }

    .badge-optional {
        background: linear-gradient(135deg, #6c757d 0%, #5a6268 100%);
        color: white;
        padding: 0.6rem 1.2rem;
        border-radius: 25px;
        font-size: 0.85rem;
        font-weight: 600;
        letter-spacing: 0.5px;
        box-shadow: 0 4px 15px rgba(108, 117, 125, 0.3);
    }

    .fields-container-spectacular {
        background: #f8f9fa;
        border-radius: 15px;
        padding: 1.5rem;
        border: 1px solid #e9ecef;
    }

    .field-spectacular {
        display: flex;
        align-items: center;
        margin-bottom: 1rem;
        padding: 1rem;
        background: white;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }

    .field-spectacular:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 25px rgba(0,0,0,0.1);
    }

    .field-spectacular:last-child {
        margin-bottom: 0;
    }

    .field-glow {
        position: absolute;
        top: 0;
        left: -100%;
        width: 100%;
        height: 100%;
        background: linear-gradient(90deg, transparent, rgba(255,255,255,0.8), transparent);
        transition: left 0.5s ease;
    }

    .field-spectacular:hover .field-glow {
        left: 100%;
    }

    .field-code-spectacular {
        background: linear-gradient(135deg, #e9ecef 0%, #dee2e6 100%);
        color: #495057;
        padding: 0.5rem 1rem;
        border-radius: 8px;
        font-size: 0.9rem;
        font-family: 'Courier New', monospace;
        margin-right: 1rem;
        min-width: 90px;
        text-align: center;
        font-weight: 700;
        border: 2px solid #dee2e6;
        position: relative;
        z-index: 1;
    }

    .required-field .field-code-spectacular {
        background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
        border-color: #f5c6cb;
        color: #721c24;
    }

    .optional-field .field-code-spectacular {
        background: linear-gradient(135deg, #d1ecf1 0%, #bee5eb 100%);
        border-color: #bee5eb;
        color: #0c5460;
    }

    .field-description {
        color: #6c757d;
        font-size: 0.95rem;
        font-weight: 500;
        position: relative;
        z-index: 1;
    }

    /* Card Footers */
    .card-footer-spectacular {
        margin-top: 2rem;
        padding-top: 1.5rem;
        border-top: 2px solid #f8f9fa;
    }

    .success-badge {
        background: linear-gradient(135deg, rgba(40, 167, 69, 0.1) 0%, rgba(25, 135, 84, 0.1) 100%);
        color: #155724;
        padding: 1rem 1.5rem;
        border-radius: 12px;
        border-left: 4px solid #28a745;
        font-weight: 600;
        display: flex;
        align-items: center;
    }

    .info-badge {
        background: linear-gradient(135deg, rgba(255, 193, 7, 0.1) 0%, rgba(255, 193, 7, 0.1) 100%);
        color: #856404;
        padding: 1rem 1.5rem;
        border-radius: 12px;
        border-left: 4px solid #ffc107;
        font-weight: 600;
        display: flex;
        align-items: center;
    }

    /* Responsive Design */
    @media (max-width: 768px) {
        .csv-guidelines-spectacular {
            padding: 2rem 1rem;
        }
        
        .title-gradient {
            font-size: 2rem;
        }
        
        .spectacular-card {
            padding: 2rem 1.5rem;
            margin-bottom: 2rem;
        }
        
        .card-header-spectacular {
            flex-direction: column;
            text-align: center;
            margin-bottom: 1.5rem;
        }
        
        .card-icon-spectacular {
            margin-right: 0;
            margin-bottom: 1rem;
        }
        
        .transformation-item {
            flex-direction: column;
            gap: 1rem;
            padding: 1rem;
        }
        
        .transformation-arrow {
            transform: rotate(90deg);
            margin: 0.5rem 0;
        }
        
        .phone-before-spectacular,
        .phone-after-spectacular {
            width: 100%;
        }
        
        .field-spectacular {
            flex-direction: column;
            align-items: flex-start;
            gap: 0.5rem;
        }
        
        .field-code-spectacular {
            margin-right: 0;
            min-width: auto;
        }
        
        .floating-shape {
            display: none;
        }
    }

    /* Additional Animations */
    .spectacular-card {
        animation: slideInUp 0.8s ease-out;
    }

    .spectacular-card:nth-child(1) {
        animation-delay: 0.2s;
    }

    .spectacular-card:nth-child(2) {
        animation-delay: 0.4s;
    }

    @keyframes slideInUp {
        0% {
            opacity: 0;
            transform: translateY(50px);
        }
        100% {
            opacity: 1;
            transform: translateY(0);
        }
    }

    /* Responsive Guidelines */
    @media (max-width: 768px) {
        .guidelines-section {
            padding: 1.5rem;
        }
        
        .guideline-card {
            padding: 1rem;
            margin-bottom: 1rem;
        }
        
        .phone-example-modern {
            flex-direction: column;
            gap: 0.5rem;
        }
        
        .phone-arrow {
            transform: rotate(90deg);
            margin: 0.5rem 0;
        }
        
        .phone-input,
        .phone-output {
            width: 100%;
        }
    }

    /* Quick Tips Section */
    .quick-tips-card {
        background: rgba(255,255,255,0.1);
        border-radius: 15px;
        padding: 1.5rem;
        backdrop-filter: blur(10px);
        border: 1px solid rgba(255,255,255,0.2);
        position: relative;
        z-index: 2;
    }

    .tip-item {
        display: flex;
        align-items: center;
        padding: 0.5rem 0;
    }

    .tip-item i {
        flex-shrink: 0;
    }

    /* Form Controls */
    .form-control {
        border-radius: 10px;
        border: 2px solid #e9ecef;
        transition: all 0.3s ease;
    }

    .form-control:focus {
        border-color: var(--whatsapp-green);
        box-shadow: 0 0 0 0.2rem rgba(37, 211, 102, 0.25);
    }

    .custom-file-label {
        border-radius: 10px;
        border: 2px solid #e9ecef;
    }

    .custom-file-input:focus~.custom-file-label {
        border-color: var(--whatsapp-green);
        box-shadow: 0 0 0 0.2rem rgba(37, 211, 102, 0.25);
    }

    /* Table Styles */
    .table {
        border-radius: 10px;
        overflow: hidden;
    }

    .table thead th {
        background: linear-gradient(135deg, var(--whatsapp-dark-green) 0%, var(--whatsapp-green) 100%);
        color: white;
        border: none;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .table tbody tr {
        transition: all 0.3s ease;
    }

    .table tbody tr:hover {
        background-color: rgba(37, 211, 102, 0.1);
        transform: scale(1.01);
    }

    /* Modal Styles */
    .modal-content {
        border-radius: 20px;
        overflow: hidden;
    }

    .modal-header {
        background: linear-gradient(135deg, var(--whatsapp-green) 0%, var(--whatsapp-dark-green) 100%);
    }

    /* Loading States */
    .btn:disabled {
        opacity: 0.7;
        cursor: not-allowed;
    }

    /* Success Animations */
    @keyframes checkmark {
        0% {
            transform: scale(0);
        }
        50% {
            transform: scale(1.2);
        }
        100% {
            transform: scale(1);
        }
    }

    .fa-check {
        animation: checkmark 0.5s ease-in-out;
    }

    /* Utility Classes */
    .rounded-lg {
        border-radius: 15px !important;
    }

    .min-vh-25 {
        min-height: 25vh;
    }

    .opacity-75 {
        opacity: 0.75;
    }

    /* Badge Enhancements */
    .badge-pill {
        border-radius: 50px;
        font-weight: 500;
        letter-spacing: 0.5px;
    }

    /* Backdrop Blur Effect */
    .backdrop-blur {
        backdrop-filter: blur(10px);
        -webkit-backdrop-filter: blur(10px);
    }
</style>
@endpushrent;
        background-clip: text;
    }

    /* Custom Scrollbar */
    ::-webkit-scrollbar {
        width: 8px;
    }

    ::-webkit-scrollbar-track {
        background: #f1f1f1;
        border-radius: 10px;
    }

    ::-webkit-scrollbar-thumb {
        background: linear-gradient(135deg, var(--whatsapp-green) 0%, var(--whatsapp-dark-green) 100%);
        border-radius: 10px;
    }

    ::-webkit-scrollbar-thumb:hover {
        background: linear-gradient(135deg, var(--whatsapp-dark-green) 0%, var(--whatsapp-green) 100%);
    }
</style>
@endpush
