<?php

namespace Modules\Contacts\Services\CrmIntegration;

use Modules\Contacts\Services\CrmIntegration\Connectors\HubSpotConnector;
use Modules\Contacts\Services\CrmIntegration\Connectors\SalesforceConnector;
use Modules\Contacts\Services\CrmIntegration\Connectors\PipedriveConnector;
use Modules\Contacts\Services\CrmIntegration\Connectors\OdooConnector;

class CrmPluginRegistry
{
    /**
     * Registered CRM connectors.
     *
     * @var array
     */
    protected $connectors = [];

    /**
     * Create a new CRM plugin registry.
     */
    public function __construct()
    {
        $this->registerDefaultConnectors();
    }

    /**
     * Register a CRM connector.
     *
     * @param string $type
     * @param CrmConnectorInterface $connector
     * @return void
     */
    public function register(string $type, CrmConnectorInterface $connector): void
    {
        $this->connectors[$type] = $connector;
    }

    /**
     * Get a CRM connector by type.
     *
     * @param string $type
     * @return CrmConnectorInterface
     * @throws \InvalidArgumentException
     */
    public function get(string $type): CrmConnectorInterface
    {
        if (!isset($this->connectors[$type])) {
            throw new \InvalidArgumentException("CRM connector '{$type}' not found.");
        }

        return $this->connectors[$type];
    }

    /**
     * Get all registered CRM connectors.
     *
     * @return array
     */
    public function getAll(): array
    {
        return $this->connectors;
    }

    /**
     * Get all available CRM types.
     *
     * @return array
     */
    public function getAvailableTypes(): array
    {
        return array_keys($this->connectors);
    }

    /**
     * Check if a CRM type is registered.
     *
     * @param string $type
     * @return bool
     */
    public function has(string $type): bool
    {
        return isset($this->connectors[$type]);
    }

    /**
     * Get CRM connector information.
     *
     * @param string $type
     * @return array
     */
    public function getConnectorInfo(string $type): array
    {
        $connector = $this->get($type);
        
        return [
            'type' => $connector->getType(),
            'display_name' => $connector->getDisplayName(),
            'required_credentials' => $connector->getRequiredCredentials(),
        ];
    }

    /**
     * Get all connector information.
     *
     * @return array
     */
    public function getAllConnectorInfo(): array
    {
        $info = [];
        
        foreach ($this->connectors as $type => $connector) {
            $info[$type] = $this->getConnectorInfo($type);
        }
        
        return $info;
    }

    /**
     * Register default CRM connectors.
     *
     * @return void
     */
    protected function registerDefaultConnectors(): void
    {
        $this->register('hubspot', new HubSpotConnector());
        $this->register('salesforce', new SalesforceConnector());
        $this->register('pipedrive', new PipedriveConnector());
        $this->register('odoo', new OdooConnector());
        
        // Future connectors can be registered here
        // $this->register('zoho', new ZohoConnector());
        // $this->register('monday', new MondayConnector());
        // $this->register('airtable', new AirtableConnector());
    }
}