<?php

namespace Modules\Contacts\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use InvalidArgumentException;

/**
 * Phone Normalization Service
 * 
 * Handles phone number normalization to international format
 * following E.164 standard for enterprise applications.
 * 
 * @package Modules\Contacts\Services
 * @author WA-Tawasul Team
 * @version 1.0.0
 */
class PhoneNormalizationService
{
    /**
     * Cache key for country codes
     */
    private const COUNTRY_CODES_CACHE_KEY = 'phone_normalization_country_codes';
    
    /**
     * Cache duration in minutes
     */
    private const CACHE_DURATION = 60 * 24; // 24 hours
    
    /**
     * Minimum phone number length
     */
    private const MIN_PHONE_LENGTH = 7;
    
    /**
     * Maximum phone number length
     */
    private const MAX_PHONE_LENGTH = 15;

    /**
     * Normalize phone number to international format without spaces
     * Handles various formats: +971 55 869 4287, 97155 8694287, etc.
     * 
     * @param string $phone The phone number to normalize
     * @return string Normalized phone number in E.164 format
     * @throws InvalidArgumentException If phone number is invalid
     */
    public function normalizePhone(string $phone): string
    {
        if (empty($phone)) {
            throw new InvalidArgumentException('Phone number cannot be empty');
        }

        // Remove all spaces, dashes, parentheses, and dots
        $phone = preg_replace('/[\s\-\(\)\.]/', '', $phone);
        
        // Remove any non-digit characters except +
        $phone = preg_replace('/[^\d\+]/', '', $phone);
        
        // Handle different starting patterns
        if (str_starts_with($phone, '+')) {
            // Already has + prefix
            return $this->validateAndFormat($phone);
        } elseif (preg_match('/^(\d{1,4})(\d{6,14})$/', $phone, $matches)) {
            // Format like 971558694287 or 1234567890
            $countryCode = $matches[1];
            $number = $matches[2];
            
            if ($this->isValidCountryCode($countryCode)) {
                return $this->validateAndFormat('+' . $countryCode . $number);
            }
        } elseif (preg_match('/^(\d{10})$/', $phone)) {
            // US format without country code
            return $this->validateAndFormat('+1' . $phone);
        } elseif (preg_match('/^(\d{11})$/', $phone) && str_starts_with($phone, '1')) {
            // US format with country code
            return $this->validateAndFormat('+' . $phone);
        }
        
        // If no pattern matches, try to add + if it's all digits
        if (preg_match('/^\d+$/', $phone)) {
            return $this->validateAndFormat('+' . $phone);
        }
        
        throw new InvalidArgumentException("Unable to normalize phone number: {$phone}");
    }
    
    /**
     * Validate and format phone number
     * 
     * @param string $phone
     * @return string
     * @throws InvalidArgumentException
     */
    private function validateAndFormat(string $phone): string
    {
        // Remove + for length validation
        $digits = substr($phone, 1);
        
        if (strlen($digits) < self::MIN_PHONE_LENGTH || strlen($digits) > self::MAX_PHONE_LENGTH) {
            throw new InvalidArgumentException("Phone number length invalid: {$phone}");
        }
        
        return $phone;
    }
    
    /**
     * Check if country code is valid
     * 
     * @param string $countryCode
     * @return bool
     */
    private function isValidCountryCode(string $countryCode): bool
    {
        $countryCodes = $this->getCountryCodes();
        return isset($countryCodes[$countryCode]);
    }
    
    /**
     * Get country codes with caching
     * 
     * @return array
     */
    private function getCountryCodes(): array
    {
        return Cache::remember(self::COUNTRY_CODES_CACHE_KEY, self::CACHE_DURATION, function () {
            return [
                '971' => '971', // UAE
                '966' => '966', // Saudi Arabia
                '965' => '965', // Kuwait
                '973' => '973', // Bahrain
                '974' => '974', // Qatar
                '968' => '968', // Oman
                '1' => '1',     // US/Canada
                '44' => '44',   // UK
                '33' => '33',   // France
                '49' => '49',   // Germany
                '91' => '91',   // India
                '86' => '86',   // China
                '81' => '81',   // Japan
                '82' => '82',   // South Korea
                '61' => '61',   // Australia
                '55' => '55',   // Brazil
                '52' => '52',   // Mexico
                '54' => '54',   // Argentina
                '56' => '56',   // Chile
                '57' => '57',   // Colombia
                '58' => '58',   // Venezuela
                '51' => '51',   // Peru
                '593' => '593', // Ecuador
                '591' => '591', // Bolivia
                '598' => '598', // Uruguay
                '595' => '595', // Paraguay
                '27' => '27',   // South Africa
                '234' => '234', // Nigeria
                '254' => '254', // Kenya
                '20' => '20',   // Egypt
                '212' => '212', // Morocco
                '213' => '213', // Algeria
                '216' => '216', // Tunisia
                '218' => '218', // Libya
                '220' => '220', // Gambia
                '221' => '221', // Senegal
                '222' => '222', // Mauritania
                '223' => '223', // Mali
                '224' => '224', // Guinea
                '225' => '225', // Ivory Coast
                '226' => '226', // Burkina Faso
                '227' => '227', // Niger
                '228' => '228', // Togo
                '229' => '229', // Benin
                '230' => '230', // Mauritius
                '231' => '231', // Liberia
                '232' => '232', // Sierra Leone
                '233' => '233', // Ghana
                '235' => '235', // Chad
                '236' => '236', // Central African Republic
                '237' => '237', // Cameroon
                '238' => '238', // Cape Verde
                '239' => '239', // São Tomé and Príncipe
                '240' => '240', // Equatorial Guinea
                '241' => '241', // Gabon
                '242' => '242', // Republic of the Congo
                '243' => '243', // Democratic Republic of the Congo
                '244' => '244', // Angola
                '245' => '245', // Guinea-Bissau
                '246' => '246', // British Indian Ocean Territory
                '248' => '248', // Seychelles
                '249' => '249', // Sudan
                '250' => '250', // Rwanda
                '251' => '251', // Ethiopia
                '252' => '252', // Somalia
                '253' => '253', // Djibouti
                '255' => '255', // Tanzania
                '256' => '256', // Uganda
                '257' => '257', // Burundi
                '258' => '258', // Mozambique
                '260' => '260', // Zambia
                '261' => '261', // Madagascar
                '262' => '262', // Mayotte
                '263' => '263', // Zimbabwe
                '264' => '264', // Namibia
                '265' => '265', // Malawi
                '266' => '266', // Lesotho
                '267' => '267', // Botswana
                '268' => '268', // Swaziland
                '269' => '269', // Comoros
                '290' => '290', // Saint Helena
                '291' => '291', // Eritrea
                '297' => '297', // Aruba
                '298' => '298', // Faroe Islands
                '299' => '299', // Greenland
            ];
        });
    }
    
    /**
     * Validate if phone number is in correct international format
     * 
     * @param string $phone
     * @return bool
     */
    public function isValidPhone(string $phone): bool
    {
        try {
            $normalized = $this->normalizePhone($phone);
            return preg_match('/^\+\d{7,15}$/', $normalized);
        } catch (InvalidArgumentException $e) {
            Log::warning('Invalid phone number validation failed', [
                'phone' => $phone,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
    
    /**
     * Get country code from phone number
     * 
     * @param string $phone
     * @return string|null
     */
    public function getCountryCode(string $phone): ?string
    {
        try {
            $normalized = $this->normalizePhone($phone);
            
            if (preg_match('/^\+(\d{1,4})/', $normalized, $matches)) {
                return $matches[1];
            }
        } catch (InvalidArgumentException $e) {
            Log::warning('Failed to extract country code', [
                'phone' => $phone,
                'error' => $e->getMessage()
            ]);
        }
        
        return null;
    }
    
    /**
     * Format phone number for display (with spaces)
     * 
     * @param string $phone
     * @return string
     */
    public function formatForDisplay(string $phone): string
    {
        try {
            $normalized = $this->normalizePhone($phone);
            
            if (preg_match('/^\+(\d{1,4})(\d{3})(\d{3})(\d{4})$/', $normalized, $matches)) {
                return '+' . $matches[1] . ' ' . $matches[2] . ' ' . $matches[3] . ' ' . $matches[4];
            } elseif (preg_match('/^\+(\d{1,4})(\d{2})(\d{3})(\d{4})$/', $normalized, $matches)) {
                return '+' . $matches[1] . ' ' . $matches[2] . ' ' . $matches[3] . ' ' . $matches[4];
            }
            
            return $normalized;
        } catch (InvalidArgumentException $e) {
            return $phone; // Return original if normalization fails
        }
    }
    
    /**
     * Batch normalize multiple phone numbers
     * 
     * @param array $phones
     * @return array
     */
    public function normalizeBatch(array $phones): array
    {
        $result = [];
        
        foreach ($phones as $index => $phone) {
            try {
                $normalized = $this->normalizePhone($phone);
                $result[$index] = [
                    'original' => $phone,
                    'normalized' => $normalized,
                    'is_valid' => true,
                    'country_code' => $this->getCountryCode($phone),
                    'error' => null
                ];
            } catch (InvalidArgumentException $e) {
                $result[$index] = [
                    'original' => $phone,
                    'normalized' => null,
                    'is_valid' => false,
                    'country_code' => null,
                    'error' => $e->getMessage()
                ];
                
                Log::warning('Phone normalization failed in batch', [
                    'phone' => $phone,
                    'error' => $e->getMessage()
                ]);
            }
        }
        
        return $result;
    }
    
    /**
     * Get phone number variations for duplicate detection
     * 
     * @param string $phone
     * @return array
     */
    public function getVariations(string $phone): array
    {
        $variations = [];
        
        try {
            // Add original phone
            $variations[] = $phone;
            
            // Add normalized version
            $normalized = $this->normalizePhone($phone);
            $variations[] = $normalized;
            
            // Add without + prefix
            if (str_starts_with($normalized, '+')) {
                $variations[] = substr($normalized, 1);
            }
            
            // Add with spaces for display format
            $variations[] = $this->formatForDisplay($phone);
            
            // Remove duplicates
            return array_unique($variations);
            
        } catch (InvalidArgumentException $e) {
            // If normalization fails, just return original
            return [$phone];
        }
    }
    
    /**
     * Clear country codes cache
     * 
     * @return void
     */
    public function clearCache(): void
    {
        Cache::forget(self::COUNTRY_CODES_CACHE_KEY);
    }
}