<?php

namespace Modules\Contacts\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;

class UpdateContactRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $companyId = session('company_id');
        $contactId = $this->route('contact');
        
        return [
            'name' => 'required|string|max:255',
            'phone' => [
                'required',
                'string',
                function ($attribute, $value, $fail) use ($companyId, $contactId) {
                    $this->validatePhoneUniqueness($attribute, $value, $fail, $companyId, $contactId);
                }
            ],
            //'email' => 'nullable|email|max:255',
            //'country_id' => 'required|exists:countries,id',
            //'groups' => 'nullable|array',
            //'groups.*' => 'exists:groups,id',
            //'custom' => 'nullable|array',
            //'avatar' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ];
    }

    /**
     * Custom validation for phone number uniqueness with normalization
     */
    private function validatePhoneUniqueness($attribute, $value, $fail, $companyId, $contactId)
    {
        $normalizedPhone = $this->normalizePhoneNumber($value);
        
        $exists = DB::table('contacts')
            ->where('company_id', $companyId)
            ->where('id', '!=', $contactId)
            ->whereNull('deleted_at')
            ->where(function($query) use ($value, $normalizedPhone) {
                $query->where('phone', $value)
                      ->orWhere('phone', $normalizedPhone)
                      ->orWhere('phone', ltrim($normalizedPhone, '+'))
                      ->orWhere('phone', '+' . ltrim($value, '+'));
            })
            ->exists();
            
        if ($exists) {
            $fail(__('contacts::general.phone_already_exists'));
        }
    }

    /**
     * Normalize phone number for consistent duplicate checking
     */
    private function normalizePhoneNumber($phone)
    {
        // Remove + and any non-digit characters
        $cleaned = preg_replace('/[^0-9]/', '', $phone);
        
        // Add + back to the beginning
        return '+' . $cleaned;
    }

    /**
     * Get custom error messages for validation rules.
     *
     * @return array
     */
    public function messages()
    {
        return [
            'phone.unique' => __('contacts::general.phone_already_exists'),
            'name.required' => __('contacts::general.name_required'),
            'phone.required' => __('contacts::general.phone_required'),
            'email.email' => __('contacts::general.email_invalid'),
            'country_id.required' => __('contacts::general.country_required'),
            'country_id.exists' => __('contacts::general.country_exists'),
            'avatar.image' => __('contacts::general.avatar_invalid'),
            'avatar.mimes' => __('contacts::general.avatar_mimes'),
            'avatar.max' => __('contacts::general.avatar_max_size')
        ];
    }
}
